"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopDependencySpans = getTopDependencySpans;
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _event_outcome = require("../../../common/event_outcome");
var _environment_query = require("../../../common/utils/environment_query");
var _maybe = require("../../../common/utils/maybe");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_NUM_SPANS = 1000;
async function getTopDependencySpans({
  apmEventClient,
  dependencyName,
  spanName,
  start,
  end,
  environment,
  kuery,
  sampleRangeFrom,
  sampleRangeTo
}) {
  const topDedsRequiredFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.TRACE_ID, _apm.SPAN_NAME, _apm.SERVICE_NAME, _apm.AGENT_NAME, _apm.SPAN_DURATION, _apm.EVENT_OUTCOME, _apm.AT_TIMESTAMP]);
  const spans = (await apmEventClient.search('get_top_dependency_spans', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: MAX_NUM_SPANS,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), ...(0, _server.termQuery)(_apm.SPAN_DESTINATION_SERVICE_RESOURCE, dependencyName), ...(0, _server.termQuery)(_apm.SPAN_NAME, spanName), ...((sampleRangeFrom !== null && sampleRangeFrom !== void 0 ? sampleRangeFrom : 0) >= 0 && (sampleRangeTo !== null && sampleRangeTo !== void 0 ? sampleRangeTo : 0) > 0 ? [{
          range: {
            [_apm.SPAN_DURATION]: {
              gte: sampleRangeFrom,
              lte: sampleRangeTo
            }
          }
        }] : [])]
      }
    },
    fields: topDedsRequiredFields
  })).hits.hits.map(hit => (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(topDedsRequiredFields));
  const traceIds = spans.map(span => span[_apm.TRACE_ID]);
  const txRequiredFields = (0, _as_mutable_array.asMutableArray)([_apm.TRACE_ID, _apm.TRANSACTION_ID, _apm.TRANSACTION_TYPE, _apm.TRANSACTION_NAME]);
  const transactions = (await apmEventClient.search('get_transactions_for_dependency_spans', {
    apm: {
      events: [_common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: traceIds.length,
    query: {
      bool: {
        filter: [...(0, _server.termsQuery)(_apm.TRACE_ID, ...traceIds), {
          exists: {
            field: _apm.TRANSACTION_ID
          }
        }]
      }
    },
    fields: txRequiredFields,
    sort: {
      '@timestamp': 'desc'
    }
  })).hits.hits.map(hit => (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(txRequiredFields));
  const transactionsByTraceId = (0, _lodash.keyBy)(transactions, transaction => transaction[_apm.TRACE_ID]);
  return spans.map(span => {
    const traceId = span[_apm.TRACE_ID];
    const transaction = (0, _maybe.maybe)(transactionsByTraceId[traceId]);
    return {
      [_apm.AT_TIMESTAMP]: new Date(span[_apm.AT_TIMESTAMP]).getTime(),
      spanId: span[_apm.SPAN_ID],
      spanName: span[_apm.SPAN_NAME],
      serviceName: span[_apm.SERVICE_NAME],
      agentName: span[_apm.AGENT_NAME],
      duration: span[_apm.SPAN_DURATION],
      traceId,
      outcome: span[_apm.EVENT_OUTCOME] || _event_outcome.EventOutcome.unknown,
      transactionId: transaction === null || transaction === void 0 ? void 0 : transaction[_apm.TRANSACTION_ID],
      transactionType: transaction === null || transaction === void 0 ? void 0 : transaction[_apm.TRANSACTION_TYPE],
      transactionName: transaction === null || transaction === void 0 ? void 0 : transaction[_apm.TRANSACTION_NAME]
    };
  });
}