"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRule = updateRule;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _lib = require("../../../../lib");
var _validate_authorize_system_actions = require("../../../../lib/validate_authorize_system_actions");
var _authorization = require("../../../../authorization");
var _common = require("../../../../../common");
var _mapped_params_utils = require("../../../../rules_client/common/mapped_params_utils");
var _retry_if_conflicts = require("../../../../lib/retry_if_conflicts");
var _bulk_mark_api_keys_for_invalidation = require("../../../../invalidate_pending_api_keys/bulk_mark_api_keys_for_invalidation");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _lib2 = require("../../../../rules_client/lib");
var _rule = require("../../../../data/rule");
var _get_schedule_frequency = require("../get_schedule_frequency");
var _saved_objects = require("../../../../saved_objects");
var _schemas = require("./schemas");
var _transforms = require("../../transforms");
var _schemas2 = require("../../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function updateRule(context, updateParams) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `rulesClient.update('${updateParams.id}')`, async () => await updateWithOCC(context, updateParams));
}
async function updateWithOCC(context, updateParams) {
  var _context$auditLogger2;
  const {
    data: initialData,
    allowMissingConnectorSecrets,
    id,
    shouldIncrementRevision = () => true
  } = updateParams;

  // Validate update rule data schema
  try {
    _schemas.updateRuleDataSchema.validate(initialData);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating update data - ${error.message}`);
  }
  let originalRuleSavedObject;
  try {
    originalRuleSavedObject = await (0, _rule.getDecryptedRuleSo)({
      id,
      encryptedSavedObjectsClient: context.encryptedSavedObjectsClient,
      savedObjectsGetOptions: {
        namespace: context.namespace
      }
    });
  } catch (e) {
    // We'll skip invalidating the API key since we failed to load the decrypted saved object
    context.logger.error(`update(): Failed to load API key to invalidate on alert ${id}: ${e.message}`);
    // Still attempt to load the object using SOC
    originalRuleSavedObject = await (0, _rule.getRuleSo)({
      id,
      savedObjectsClient: context.unsecuredSavedObjectsClient
    });
  }
  const {
    actions: genActions,
    systemActions: genSystemActions
  } = await (0, _lib2.addGeneratedActionValues)(initialData.actions, initialData.systemActions, context);
  const data = {
    ...initialData,
    actions: genActions,
    systemActions: genSystemActions
  };
  const {
    alertTypeId,
    consumer,
    enabled,
    schedule,
    name,
    apiKey,
    apiKeyCreatedByUser
  } = originalRuleSavedObject.attributes;
  let validationPayload = null;
  if (enabled && schedule.interval !== data.schedule.interval) {
    validationPayload = await (0, _get_schedule_frequency.validateScheduleLimit)({
      context,
      prevInterval: schedule === null || schedule === void 0 ? void 0 : schedule.interval,
      updatedInterval: data.schedule.interval
    });
  }
  if (validationPayload) {
    throw _boom.default.badRequest((0, _common.getRuleCircuitBreakerErrorMessage)({
      name,
      interval: validationPayload.interval,
      intervalAvailable: validationPayload.intervalAvailable,
      action: 'update'
    }));
  }
  try {
    await context.authorization.ensureAuthorized({
      ruleTypeId: alertTypeId,
      consumer,
      operation: _authorization.WriteOperations.Update,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.UPDATE,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id,
        name
      },
      error
    }));
    throw error;
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.UPDATE,
    outcome: 'unknown',
    savedObject: {
      type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
      id,
      name
    }
  }));
  context.ruleTypeRegistry.ensureRuleTypeEnabled(originalRuleSavedObject.attributes.alertTypeId);
  const ruleType = context.ruleTypeRegistry.get(originalRuleSavedObject.attributes.alertTypeId);

  // Validate Rule types and actions
  const actionsClient = await context.getActionsClient();
  const validatedRuleTypeParams = (0, _lib.validateRuleTypeParams)(data.params, ruleType.validate.params);
  await (0, _lib2.validateActions)(context, ruleType, data, allowMissingConnectorSecrets);
  await (0, _validate_authorize_system_actions.validateAndAuthorizeSystemActions)({
    actionsClient,
    actionsAuthorization: context.actionsAuthorization,
    connectorAdapterRegistry: context.connectorAdapterRegistry,
    systemActions: data.systemActions,
    rule: {
      consumer: originalRuleSavedObject.attributes.consumer,
      producer: ruleType.producer
    }
  });

  // Throw error if schedule interval is less than the minimum and we are enforcing it
  const intervalInMs = (0, _common.parseDuration)(data.schedule.interval);
  if (intervalInMs < context.minimumScheduleIntervalInMs && context.minimumScheduleInterval.enforce) {
    throw _boom.default.badRequest(`Error updating rule: the interval is less than the allowed minimum interval of ${context.minimumScheduleInterval.value}`);
  }
  const updateResult = await updateRuleAttributes({
    context,
    updateRuleData: data,
    validatedRuleTypeParams: validatedRuleTypeParams,
    originalRuleSavedObject,
    shouldIncrementRevision,
    isSystemAction: connectorId => actionsClient.isSystemAction(connectorId)
  });

  // Log warning if schedule interval is less than the minimum but we're not enforcing it
  if (intervalInMs < context.minimumScheduleIntervalInMs && !context.minimumScheduleInterval.enforce) {
    context.logger.warn(`Rule schedule interval (${data.schedule.interval}) for "${ruleType.id}" rule type with ID "${id}" is less than the minimum value (${context.minimumScheduleInterval.value}). Running rules at this interval may impact alerting performance. Set "xpack.alerting.rules.minimumScheduleInterval.enforce" to true to prevent such changes.`);
  }
  await Promise.all([apiKey && !apiKeyCreatedByUser ? (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
    apiKeys: [apiKey]
  }, context.logger, context.unsecuredSavedObjectsClient) : null, (async () => {
    if (updateResult.scheduledTaskId && updateResult.schedule && !(0, _lodash.isEqual)(schedule, updateResult.schedule)) {
      try {
        var _tasks$;
        const {
          tasks
        } = await context.taskManager.bulkUpdateSchedules([updateResult.scheduledTaskId], updateResult.schedule);
        context.logger.debug(`Rule update has rescheduled the underlying task: ${updateResult.scheduledTaskId} to run at: ${tasks === null || tasks === void 0 ? void 0 : (_tasks$ = tasks[0]) === null || _tasks$ === void 0 ? void 0 : _tasks$.runAt}`);
      } catch (err) {
        context.logger.error(`Rule update failed to run its underlying task. TaskManager bulkUpdateSchedules failed with Error: ${err.message}`);
      }
    }
  })()]);
  return updateResult;
}
async function updateRuleAttributes({
  context,
  updateRuleData,
  validatedRuleTypeParams,
  originalRuleSavedObject,
  shouldIncrementRevision,
  isSystemAction
}) {
  var _updateRuleData$syste, _updateRuleData$artif, _updateRuleData$notif, _updateRuleData$throt;
  await (0, _lib2.bulkMigrateLegacyActions)({
    context,
    rules: [originalRuleSavedObject]
  });
  const originalRule = originalRuleSavedObject.attributes;
  const allActions = [...updateRuleData.actions, ...((_updateRuleData$syste = updateRuleData.systemActions) !== null && _updateRuleData$syste !== void 0 ? _updateRuleData$syste : [])];
  const artifacts = (_updateRuleData$artif = updateRuleData.artifacts) !== null && _updateRuleData$artif !== void 0 ? _updateRuleData$artif : {};
  const ruleType = context.ruleTypeRegistry.get(originalRule.alertTypeId);

  // Extract saved object references for this rule
  const {
    references: extractedReferences,
    params: updatedParams,
    actions: actionsWithRefs,
    artifacts: artifactsWithRefs
  } = await (0, _lib2.extractReferences)(context, ruleType, allActions, validatedRuleTypeParams, artifacts);

  // Increment revision if applicable field has changed
  const revision = shouldIncrementRevision(updatedParams) ? (0, _lib2.incrementRevision)({
    originalRule,
    updateRuleData,
    updatedParams
  }) : originalRule.revision;
  const username = await context.getUserName();
  const apiKeyAttributes = await (0, _lib2.createNewAPIKeySet)(context, {
    id: ruleType.id,
    ruleName: updateRuleData.name,
    username,
    shouldUpdateApiKey: originalRule.enabled,
    errorMessage: 'Error updating rule: could not create API key'
  });
  const notifyWhen = (0, _lib.getRuleNotifyWhenType)((_updateRuleData$notif = updateRuleData.notifyWhen) !== null && _updateRuleData$notif !== void 0 ? _updateRuleData$notif : null, (_updateRuleData$throt = updateRuleData.throttle) !== null && _updateRuleData$throt !== void 0 ? _updateRuleData$throt : null);
  const updatedRuleAttributes = (0, _lib2.updateMetaAttributes)(context, {
    ...originalRule,
    ...(0, _lodash.omit)(updateRuleData, 'actions', 'systemActions', 'artifacts'),
    ...apiKeyAttributes,
    params: updatedParams,
    actions: actionsWithRefs,
    notifyWhen,
    revision,
    updatedBy: username,
    updatedAt: new Date().toISOString(),
    artifacts: artifactsWithRefs
  });
  const mappedParams = (0, _mapped_params_utils.getMappedParams)(updatedParams);
  if (Object.keys(mappedParams).length) {
    updatedRuleAttributes.mapped_params = mappedParams;
  }
  let updatedRuleSavedObject;
  const {
    id,
    version
  } = originalRuleSavedObject;
  try {
    updatedRuleSavedObject = await (0, _rule.createRuleSo)({
      savedObjectsClient: context.unsecuredSavedObjectsClient,
      ruleAttributes: updatedRuleAttributes,
      savedObjectsCreateOptions: {
        id,
        version,
        overwrite: true,
        references: extractedReferences
      }
    });
  } catch (e) {
    // Avoid unused API key
    await (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
      apiKeys: updatedRuleAttributes.apiKey && !updatedRuleAttributes.apiKeyCreatedByUser ? [updatedRuleAttributes.apiKey] : []
    }, context.logger, context.unsecuredSavedObjectsClient);
    throw e;
  }
  const ruleDomain = (0, _transforms.transformRuleAttributesToRuleDomain)(updatedRuleSavedObject.attributes, {
    id: updatedRuleSavedObject.id,
    logger: context.logger,
    ruleType,
    references: updatedRuleSavedObject.references
  }, isSystemAction);

  // Try to validate created rule, but don't throw.
  try {
    _schemas2.ruleDomainSchema.validate(ruleDomain);
  } catch (e) {
    context.logger.warn(`Error validating updated rule domain object for id: ${id}, ${e}`);
  }

  // Convert domain rule to rule (Remove certain properties)
  const rule = (0, _transforms.transformRuleDomainToRule)(ruleDomain, {
    isPublic: true
  });

  // TODO (http-versioning): Remove this cast, this enables us to move forward
  // without fixing all of other solution types
  return rule;
}