"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEditRuleParamsWithReadAuth = bulkEditRuleParamsWithReadAuth;
var _lodash = require("lodash");
var _bulk_edit = require("../../../../rules_client/common/bulk_edit");
var _saved_objects = require("../../../../saved_objects");
var _authorization = require("../../../../authorization");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _common = require("../../../../rules_client/common");
var _schemas = require("./schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function bulkEditRuleParamsWithReadAuth(context, options) {
  try {
    _schemas.bulkEditRuleParamsOptionsSchema.validate({
      ids: options.ids,
      filter: options.filter,
      operations: options.operations
    });
  } catch (error) {
    throw new Error(`Error validating bulkEditRuleParamsWithReadAuth options - ${error.message}`);
  }
  const shouldInvalidateApiKeys = false;
  // only require read access to the rules in order to bulk edit the params
  const auditAction = _audit_events.RuleAuditAction.BULK_EDIT_PARAMS;
  const requiredAuthOperation = _authorization.ReadOperations.BulkEditParams;
  const result = await (0, _bulk_edit.bulkEditRules)(context, {
    ...options,
    name: `rulesClient.bulkEditRuleParams('operations=${JSON.stringify(options.operations)}, paramsModifier=${options.paramsModifier ? '[Function]' : undefined}', shouldIncrementRevision=${options.shouldIncrementRevision ? '[Function]' : undefined}')`,
    auditAction,
    requiredAuthOperation,
    shouldInvalidateApiKeys,
    updateFn: opts => updateRuleParamsInMemory({
      ...opts,
      context,
      shouldInvalidateApiKeys,
      operations: options.operations,
      paramsModifier: options.paramsModifier,
      shouldIncrementRevision: options.shouldIncrementRevision
    })
  });
  return result;
}
async function updateRuleParamsInMemory({
  context,
  rule,
  operations,
  apiKeysMap,
  rules,
  skipped,
  errors,
  username,
  shouldInvalidateApiKeys,
  paramsModifier,
  shouldIncrementRevision = () => true
}) {
  try {
    await (0, _bulk_edit.updateRuleInMemory)(context, {
      rule,
      apiKeysMap,
      rules,
      skipped,
      errors,
      username,
      paramsModifier,
      shouldInvalidateApiKeys,
      shouldIncrementRevision,
      updateAttributesFn: async ({
        domainRule,
        ruleActions
      }) => {
        const result = await getUpdatedParamAttributesFromOperations({
          operations,
          rule: domainRule
        });
        return {
          ...result,
          ruleActions,
          hasUpdateApiKeyOperation: false
        };
      }
    });
  } catch (error) {
    var _rule$attributes, _context$auditLogger, _rule$attributes2;
    errors.push({
      message: error.message,
      rule: {
        id: rule.id,
        name: (_rule$attributes = rule.attributes) === null || _rule$attributes === void 0 ? void 0 : _rule$attributes.name
      }
    });
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.BULK_EDIT_PARAMS,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id: rule.id,
        name: (_rule$attributes2 = rule.attributes) === null || _rule$attributes2 === void 0 ? void 0 : _rule$attributes2.name
      },
      error
    }));
  }
}
async function getUpdatedParamAttributesFromOperations({
  operations,
  rule
}) {
  let updatedRule = (0, _lodash.cloneDeep)(rule);
  let isAttributesUpdateSkipped = true;
  for (const operation of operations) {
    const updatedOperation = {
      ...operation,
      field: `params.${operation.field}`
    };
    const {
      modifiedAttributes,
      isAttributeModified
    } = (0, _common.applyBulkEditOperation)(updatedOperation, updatedRule);
    if (isAttributeModified) {
      updatedRule = {
        ...updatedRule,
        ...modifiedAttributes
      };
      isAttributesUpdateSkipped = false;
    }

    // Only increment revision if update wasn't skipped and `operation.field` should result in a revision increment
    if (!isAttributesUpdateSkipped && rule.revision - updatedRule.revision === 0) {
      updatedRule.revision += 1;
    }
  }
  return {
    rule: updatedRule,
    isAttributesUpdateSkipped
  };
}