"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RULE_GAP_AGGREGATIONS = void 0;
exports.calculateHighestPriorityGapFillStatus = calculateHighestPriorityGapFillStatus;
exports.extractGapDurationSums = extractGapDurationSums;
exports.hasMatchedGapFillStatus = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Extracts and normalizes gap duration sums from an aggregation bucket
 */
function extractGapDurationSums(bucket) {
  var _bucket$totalUnfilled, _bucket$totalUnfilled2, _bucket$totalInProgre, _bucket$totalInProgre2, _bucket$totalFilledDu, _bucket$totalFilledDu2, _bucket$totalDuration, _bucket$totalDuration2;
  return {
    totalUnfilledDurationMs: Math.max(0, (_bucket$totalUnfilled = (_bucket$totalUnfilled2 = bucket.totalUnfilledDurationMs) === null || _bucket$totalUnfilled2 === void 0 ? void 0 : _bucket$totalUnfilled2.value) !== null && _bucket$totalUnfilled !== void 0 ? _bucket$totalUnfilled : 0),
    totalInProgressDurationMs: Math.max(0, (_bucket$totalInProgre = (_bucket$totalInProgre2 = bucket.totalInProgressDurationMs) === null || _bucket$totalInProgre2 === void 0 ? void 0 : _bucket$totalInProgre2.value) !== null && _bucket$totalInProgre !== void 0 ? _bucket$totalInProgre : 0),
    totalFilledDurationMs: Math.max(0, (_bucket$totalFilledDu = (_bucket$totalFilledDu2 = bucket.totalFilledDurationMs) === null || _bucket$totalFilledDu2 === void 0 ? void 0 : _bucket$totalFilledDu2.value) !== null && _bucket$totalFilledDu !== void 0 ? _bucket$totalFilledDu : 0),
    totalDurationMs: Math.max(0, (_bucket$totalDuration = (_bucket$totalDuration2 = bucket.totalDurationMs) === null || _bucket$totalDuration2 === void 0 ? void 0 : _bucket$totalDuration2.value) !== null && _bucket$totalDuration !== void 0 ? _bucket$totalDuration : 0)
  };
}

/**
 * Calculates aggregated gap fill status based on duration sums
 * Precedence: unfilled > in_progress > filled
 */
function calculateHighestPriorityGapFillStatus(sums) {
  const {
    totalInProgressDurationMs,
    totalUnfilledDurationMs,
    totalFilledDurationMs
  } = sums;
  if (totalUnfilledDurationMs > 0) return 'unfilled';
  if (totalInProgressDurationMs > 0) return 'in_progress';
  if (totalFilledDurationMs > 0) return 'filled';
  return null;
}

/**
 * Common aggregation fields for gap duration tracking
 */
const RULE_GAP_AGGREGATIONS = exports.RULE_GAP_AGGREGATIONS = {
  totalUnfilledDurationMs: {
    sum: {
      field: 'kibana.alert.rule.gap.unfilled_duration_ms'
    }
  },
  totalInProgressDurationMs: {
    sum: {
      field: 'kibana.alert.rule.gap.in_progress_duration_ms'
    }
  },
  totalFilledDurationMs: {
    sum: {
      field: 'kibana.alert.rule.gap.filled_duration_ms'
    }
  },
  totalDurationMs: {
    sum: {
      field: 'kibana.alert.rule.gap.total_gap_duration_ms'
    }
  }
};
/**
 * Checks if the gap fill status of the bucket matches the given gap fill statuses
 */
const hasMatchedGapFillStatus = (bucket, gapFillStatuses) => {
  const sums = extractGapDurationSums(bucket);
  const gapFillStatus = calculateHighestPriorityGapFillStatus(sums);
  return Boolean(gapFillStatus && gapFillStatuses.includes(gapFillStatus));
};
exports.hasMatchedGapFillStatus = hasMatchedGapFillStatus;