"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGapAutoFillScheduler = updateGapAutoFillScheduler;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _saved_objects = require("../../../../../saved_objects");
var _audit_events = require("../../../../../rules_client/common/audit_events");
var _authorization = require("../../../../../authorization");
var _scheduler = require("../../../types/scheduler");
var _transforms = require("../../transforms");
var _schemas = require("./schemas");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toRuleTypeKey = ruleType => `${ruleType.type}:${ruleType.consumer}`;
async function updateGapAutoFillScheduler(context, params) {
  var _schedulerSo$attribut, _schedulerSo$attribut2, _schedulerSo$attribut3, _schedulerSo$attribut4, _context$getUserName, _params$ruleTypes;
  try {
    _schemas.updateGapAutoFillSchedulerSchema.validate(params);
  } catch (error) {
    const {
      request: _req,
      ...otherParams
    } = params;
    throw _boom.default.badRequest(`Error validating gap auto fill scheduler update parameters - ${error.message}\nPayload summary: ${JSON.stringify(otherParams)}`);
  }
  const soClient = context.unsecuredSavedObjectsClient;
  const taskManager = context.taskManager;
  const schedulerSo = await (0, _utils.getGapAutoFillSchedulerSO)({
    context,
    id: params.id,
    operation: _authorization.WriteOperations.UpdateGapAutoFillScheduler,
    authAuditAction: _audit_events.GapAutoFillSchedulerAuditAction.UPDATE
  });
  const schedulerName = (_schedulerSo$attribut = (_schedulerSo$attribut2 = schedulerSo.attributes) === null || _schedulerSo$attribut2 === void 0 ? void 0 : _schedulerSo$attribut2.name) !== null && _schedulerSo$attribut !== void 0 ? _schedulerSo$attribut : params.id;
  const existingRuleTypes = (_schedulerSo$attribut3 = (_schedulerSo$attribut4 = schedulerSo.attributes) === null || _schedulerSo$attribut4 === void 0 ? void 0 : _schedulerSo$attribut4.ruleTypes) !== null && _schedulerSo$attribut3 !== void 0 ? _schedulerSo$attribut3 : [];
  const username = await ((_context$getUserName = context.getUserName) === null || _context$getUserName === void 0 ? void 0 : _context$getUserName.call(context));
  const updatedRuleTypes = (_params$ruleTypes = params.ruleTypes) !== null && _params$ruleTypes !== void 0 ? _params$ruleTypes : existingRuleTypes;
  const uniqueRuleTypeIds = new Set(updatedRuleTypes.map(({
    type
  }) => type));

  // Throw error if a rule type is not registered
  for (const ruleTypeId of uniqueRuleTypeIds) {
    context.ruleTypeRegistry.get(ruleTypeId);
  }
  const updatedRuleTypePairs = Array.from(new Set(updatedRuleTypes.map(toRuleTypeKey)));
  if (updatedRuleTypePairs.length > 0) {
    const filter = `(${updatedRuleTypePairs.map(pair => `${_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE}.attributes.ruleTypeConsumerPairs: "${pair}"`).join(' or ')})`;
    const {
      saved_objects: candidates
    } = await soClient.find({
      type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
      perPage: updatedRuleTypePairs.length + 1,
      filter
    });
    const hasConflictingPair = candidates.some(candidate => candidate.id !== params.id);
    if (hasConflictingPair) {
      throw _boom.default.conflict(`A gap auto fill scheduler already exists for at least one of the specified (rule type, consumer) pairs`);
    }
  }

  // Authorize against any newly requested rule types in the update params
  try {
    for (const ruleType of params.ruleTypes) {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType.type,
        consumer: ruleType.consumer,
        operation: _authorization.WriteOperations.UpdateGapAutoFillScheduler,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    }
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.UPDATE,
      savedObject: {
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        id: params.id,
        name: schedulerName
      },
      error: error
    }));
    throw error;
  }
  const now = new Date().toISOString();
  const updatedEnabled = params.enabled;
  const updatedSchedule = params.schedule;
  const updatedScope = params.scope !== undefined ? params.scope : schedulerSo.attributes.scope;
  const updatedAttributes = {
    name: params.name,
    enabled: updatedEnabled,
    schedule: updatedSchedule,
    gapFillRange: params.gapFillRange,
    maxBackfills: params.maxBackfills,
    numRetries: params.numRetries,
    scope: updatedScope,
    ruleTypes: updatedRuleTypes,
    ruleTypeConsumerPairs: Array.from(new Set(updatedRuleTypes.map(toRuleTypeKey))),
    updatedAt: now,
    updatedBy: username !== null && username !== void 0 ? username : undefined
  };
  try {
    var _updatedSo$attributes, _context$auditLogger2;
    await soClient.update(_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE, params.id, updatedAttributes);
    const updatedSo = await soClient.get(_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE, params.id);
    const nextSchedulerName = (_updatedSo$attributes = updatedSo.attributes.name) !== null && _updatedSo$attributes !== void 0 ? _updatedSo$attributes : schedulerName;
    await taskManager.removeIfExists(updatedSo.id);
    if (updatedEnabled) {
      await taskManager.ensureScheduled({
        id: updatedSo.id,
        taskType: _scheduler.GAP_AUTO_FILL_SCHEDULER_TASK_TYPE,
        schedule: updatedSchedule,
        scope: updatedScope !== null && updatedScope !== void 0 ? updatedScope : [],
        params: {
          configId: updatedSo.id,
          spaceId: context.spaceId
        },
        state: {}
      }, {
        request: params.request
      });
    } else {
      await context.backfillClient.deleteBackfillsByInitiatorId({
        initiatorId: updatedSo.id,
        unsecuredSavedObjectsClient: soClient,
        shouldUpdateGaps: true,
        internalSavedObjectsRepository: context.internalSavedObjectsRepository,
        eventLogClient: await context.getEventLogClient(),
        eventLogger: context.eventLogger,
        actionsClient: await context.getActionsClient()
      });
    }
    (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.UPDATE,
      savedObject: {
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        id: params.id,
        name: nextSchedulerName
      }
    }));
    return (0, _transforms.transformSavedObjectToGapAutoFillSchedulerResult)({
      savedObject: updatedSo
    });
  } catch (err) {
    var _context$auditLogger3;
    const errorMessage = `Failed to update gap auto fill scheduler by id: ${params.id}`;
    context.logger.error(`${errorMessage} - ${err}`);
    (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.UPDATE,
      savedObject: {
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        id: params.id,
        name: schedulerName
      },
      error: err
    }));
    throw _boom.default.boomify(err, {
      message: errorMessage
    });
  }
}