"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TOTAL_FIELDS_LIMIT = exports.ECS_CONTEXT = exports.ECS_COMPONENT_TEMPLATE_NAME = exports.AlertsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _alertsAsDataUtils = require("@kbn/alerts-as-data-utils");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _default_lifecycle_policy = require("./default_lifecycle_policy");
var _resource_installer_utils = require("./resource_installer_utils");
var _create_resource_installation_helper = require("./create_resource_installation_helper");
var _lib = require("./lib");
var _alerts_client = require("../alerts_client");
var _set_alerts_to_untracked = require("./lib/set_alerts_to_untracked");
var _clear_alert_flapping_history = require("./lib/clear_alert_flapping_history");
var _is_existing_alert = require("./lib/is_existing_alert");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TOTAL_FIELDS_LIMIT = exports.TOTAL_FIELDS_LIMIT = 2500;
const LEGACY_ALERT_CONTEXT = 'legacy-alert';
const ECS_CONTEXT = exports.ECS_CONTEXT = `ecs`;
const ECS_COMPONENT_TEMPLATE_NAME = exports.ECS_COMPONENT_TEMPLATE_NAME = (0, _resource_installer_utils.getComponentTemplateName)({
  name: ECS_CONTEXT
});
class AlertsService {
  constructor(options) {
    (0, _defineProperty2.default)(this, "initialized", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    (0, _defineProperty2.default)(this, "isInitializing", false);
    (0, _defineProperty2.default)(this, "resourceInitializationHelper", void 0);
    (0, _defineProperty2.default)(this, "registeredContexts", new Map());
    (0, _defineProperty2.default)(this, "commonInitPromise", void 0);
    (0, _defineProperty2.default)(this, "dataStreamAdapter", void 0);
    this.options = options;
    this.initialized = false;
    this.isServerless = options.isServerless;
    this.dataStreamAdapter = options.dataStreamAdapter;

    // Kick off initialization of common assets and save the promise
    this.commonInitPromise = this.initializeCommon(this.options.elasticsearchAndSOAvailability$, this.options.timeoutMs);

    // Create helper for initializing context-specific resources
    this.resourceInitializationHelper = (0, _create_resource_installation_helper.createResourceInstallationHelper)(this.options.logger, this.commonInitPromise, this.initializeContext.bind(this));
  }
  isInitialized() {
    return this.initialized;
  }
  async createAlertsClient(opts) {
    if (!opts.ruleType.alerts) {
      return null;
    }

    // Check if context specific installation has succeeded
    const {
      result: initialized,
      error
    } = await this.getContextInitializationPromise(opts.ruleType.alerts.context, opts.namespace);

    // If initialization failed, retry
    if (!initialized && error) {
      let initPromise;

      // If !this.initialized, we know that common resource initialization failed
      // and we need to retry this before retrying the context specific resources
      // However, if this.isInitializing, then the alerts service is in the process
      // of retrying common installation, so we don't want to kick off another retry
      if (!this.initialized) {
        if (!this.isInitializing) {
          this.options.logger.info(`Retrying common resource initialization`);
          initPromise = this.initializeCommon(this.options.elasticsearchAndSOAvailability$, this.options.timeoutMs);
        } else {
          this.options.logger.info(`Skipped retrying common resource initialization because it is already being retried.`);
        }
      }
      this.resourceInitializationHelper.retry(opts.ruleType.alerts, opts.namespace, initPromise);
      const retryResult = await this.resourceInitializationHelper.getInitializedContext(opts.ruleType.alerts.context, opts.ruleType.alerts.isSpaceAware ? opts.namespace : _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING);
      if (!retryResult.result) {
        const errorLogPrefix = `There was an error in the framework installing namespace-level resources and creating concrete indices for context "${opts.ruleType.alerts.context}" - `;
        // Retry also failed
        this.options.logger.warn(retryResult.error === error ? `${errorLogPrefix}Retry failed with error: ${error}` : `${errorLogPrefix}Original error: ${error}; Error after retry: ${retryResult.error}`);
        return null;
      } else {
        this.options.logger.info(`Resource installation for "${opts.ruleType.alerts.context}" succeeded after retry`);
      }
    }

    // TODO - when we replace the LegacyAlertsClient, we will need to decide whether to
    // initialize the AlertsClient even if alert resource installation failed. That would allow
    // us to detect alerts and trigger notifications even if we can't persist the alerts
    // (partial rule failure vs failing the entire rule execution).
    return new _alerts_client.AlertsClient({
      alertingEventLogger: opts.alertingEventLogger,
      logger: this.options.logger,
      elasticsearchClientPromise: this.options.elasticsearchClientPromise,
      ruleType: opts.ruleType,
      maintenanceWindowsService: opts.maintenanceWindowsService,
      namespace: opts.namespace,
      request: opts.request,
      rule: opts.rule,
      spaceId: opts.spaceId,
      kibanaVersion: this.options.kibanaVersion,
      dataStreamAdapter: this.dataStreamAdapter,
      isServerless: this.isServerless
    });
  }
  async getContextInitializationPromise(context, namespace) {
    const registeredOpts = this.registeredContexts.has(context) ? this.registeredContexts.get(context) : null;
    if (!registeredOpts) {
      const errMsg = `Error getting initialized status for context ${context} - context has not been registered.`;
      this.options.logger.error(errMsg);
      return (0, _create_resource_installation_helper.errorResult)(errMsg);
    }
    const result = await this.resourceInitializationHelper.getInitializedContext(context, registeredOpts.isSpaceAware ? namespace : _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING);

    // If the context is unrecognized and namespace is not the default, we
    // need to kick off resource installation and return the promise
    if (result.error && result.error.includes(`Unrecognized context`) && namespace !== _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) {
      this.resourceInitializationHelper.add(registeredOpts, namespace);
      return this.resourceInitializationHelper.getInitializedContext(context, namespace);
    }
    return result;
  }
  register(opts, timeoutMs) {
    const {
      context
    } = opts;
    // check whether this context has been registered before
    if (this.registeredContexts.has(context)) {
      const registeredOptions = this.registeredContexts.get(context);
      if (!(0, _lodash.isEqual)((0, _lodash.omit)(opts, 'shouldWrite'), (0, _lodash.omit)(registeredOptions, 'shouldWrite'))) {
        throw new Error(`${context} has already been registered with different options`);
      }
      this.options.logger.debug(`Resources for context "${context}" have already been registered.`);
      return;
    }
    this.options.logger.debug(`Registering resources for context "${context}".`);
    this.registeredContexts.set(context, opts);

    // When a context is registered, we install resources in the default namespace by default
    this.resourceInitializationHelper.add(opts, _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs);
  }

  /**
   * Initializes the common ES resources needed for framework alerts as data
   * - ILM policy - common policy shared by all AAD indices
   * - Component template - common mappings for fields populated and used by the framework
   */
  async initializeCommon(elasticsearchAndSOAvailability$, timeoutMs) {
    this.isInitializing = true;
    // Wait to install resources until ES is ready
    await (0, _rxjs.firstValueFrom)(elasticsearchAndSOAvailability$.pipe((0, _rxjs.filter)(areESAndSOAvailable => areESAndSOAvailable)));
    try {
      this.options.logger.debug(`Initializing resources for AlertsService`);
      const esClient = await this.options.elasticsearchClientPromise;

      // Common initialization installs ILM policy and shared component templates
      const initFns = [() => (0, _lib.createOrUpdateIlmPolicy)({
        logger: this.options.logger,
        esClient,
        name: _default_lifecycle_policy.DEFAULT_ALERTS_ILM_POLICY_NAME,
        policy: _default_lifecycle_policy.DEFAULT_ALERTS_ILM_POLICY,
        dataStreamAdapter: this.dataStreamAdapter
      }), () => (0, _lib.createOrUpdateComponentTemplate)({
        logger: this.options.logger,
        esClient,
        template: (0, _resource_installer_utils.getComponentTemplate)({
          fieldMap: _alertsAsDataUtils.alertFieldMap,
          includeSettings: true
        }),
        totalFieldsLimit: TOTAL_FIELDS_LIMIT
      }), () => (0, _lib.createOrUpdateComponentTemplate)({
        logger: this.options.logger,
        esClient,
        template: (0, _resource_installer_utils.getComponentTemplate)({
          fieldMap: _alertsAsDataUtils.legacyAlertFieldMap,
          name: LEGACY_ALERT_CONTEXT,
          includeSettings: true
        }),
        totalFieldsLimit: TOTAL_FIELDS_LIMIT
      }), () => (0, _lib.createOrUpdateComponentTemplate)({
        logger: this.options.logger,
        esClient,
        template: (0, _resource_installer_utils.getComponentTemplate)({
          fieldMap: _alertsAsDataUtils.ecsFieldMap,
          name: ECS_CONTEXT,
          includeSettings: true
        }),
        totalFieldsLimit: TOTAL_FIELDS_LIMIT
      })];

      // Install in parallel
      await Promise.all(initFns.map(fn => (0, _lib.installWithTimeout)({
        installFn: async () => await fn(),
        pluginStop$: this.options.pluginStop$,
        logger: this.options.logger,
        timeoutMs
      })));
      this.initialized = true;
      this.isInitializing = false;
      return (0, _create_resource_installation_helper.successResult)();
    } catch (err) {
      if (err instanceof _lib.InstallShutdownError) {
        this.options.logger.debug(err.message);
      } else {
        this.options.logger.error(`Error installing common resources for AlertsService. No additional resources will be installed and rule execution may be impacted. - ${err.message}`);
      }
      this.initialized = false;
      this.isInitializing = false;
      return (0, _create_resource_installation_helper.errorResult)(err.message);
    }
  }
  async initializeContext({
    context,
    mappings,
    useEcs,
    useLegacyAlerts,
    secondaryAlias
  }, namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs) {
    const esClient = await this.options.elasticsearchClientPromise;
    const indexTemplateAndPattern = (0, _resource_installer_utils.getIndexTemplateAndPattern)({
      context,
      namespace,
      secondaryAlias
    });
    let initFns = [];

    // List of component templates to reference
    // Order matters in this list - templates specified last take precedence over those specified first
    // 1. ECS component template, if using
    // 2. Context specific component template, if defined during registration
    // 3. Legacy alert component template, if using
    // 4. Framework common component template, always included
    const componentTemplateRefs = [];

    // If useEcs is set to true, add the ECS component template to the references
    if (useEcs) {
      componentTemplateRefs.push((0, _resource_installer_utils.getComponentTemplateName)({
        name: ECS_CONTEXT
      }));
    }

    // If fieldMap is not empty, create a context specific component template and add to the references
    if (!(0, _lodash.isEmpty)(mappings.fieldMap)) {
      const componentTemplate = (0, _resource_installer_utils.getComponentTemplate)({
        fieldMap: mappings.fieldMap,
        dynamic: mappings.dynamic,
        dynamicTemplates: mappings.dynamicTemplates,
        context
      });
      initFns.push(async () => await (0, _lib.createOrUpdateComponentTemplate)({
        logger: this.options.logger,
        esClient,
        template: componentTemplate,
        totalFieldsLimit: TOTAL_FIELDS_LIMIT
      }));
      componentTemplateRefs.push(componentTemplate.name);
    }

    // If useLegacy is set to true, add the legacy alert component template to the references
    if (useLegacyAlerts) {
      componentTemplateRefs.push((0, _resource_installer_utils.getComponentTemplateName)({
        name: LEGACY_ALERT_CONTEXT
      }));
    }

    // Add framework component template to the references
    componentTemplateRefs.push((0, _resource_installer_utils.getComponentTemplateName)());

    // Context specific initialization installs index template and write index
    initFns = initFns.concat([async () => await (0, _lib.createOrUpdateIndexTemplate)({
      logger: this.options.logger,
      esClient,
      template: (0, _lib.getIndexTemplate)({
        componentTemplateRefs,
        ilmPolicyName: _default_lifecycle_policy.DEFAULT_ALERTS_ILM_POLICY_NAME,
        indexPatterns: indexTemplateAndPattern,
        kibanaVersion: this.options.kibanaVersion,
        namespace,
        totalFieldsLimit: TOTAL_FIELDS_LIMIT,
        dataStreamAdapter: this.dataStreamAdapter
      })
    }), async () => await (0, _lib.createConcreteWriteIndex)({
      logger: this.options.logger,
      esClient,
      totalFieldsLimit: TOTAL_FIELDS_LIMIT,
      indexPatterns: indexTemplateAndPattern,
      dataStreamAdapter: this.dataStreamAdapter
    })]);

    // We want to install these in sequence and not in parallel because
    // the concrete index depends on the index template which depends on
    // the component template.
    for (const fn of initFns) {
      await (0, _lib.installWithTimeout)({
        installFn: async () => await fn(),
        pluginStop$: this.options.pluginStop$,
        logger: this.options.logger,
        timeoutMs
      });
    }
  }
  async setAlertsToUntracked(opts) {
    return (0, _set_alerts_to_untracked.setAlertsToUntracked)({
      logger: this.options.logger,
      esClient: await this.options.elasticsearchClientPromise,
      ...opts
    });
  }
  async clearAlertFlappingHistory(opts) {
    return (0, _clear_alert_flapping_history.clearAlertFlappingHistory)({
      logger: this.options.logger,
      esClient: await this.options.elasticsearchClientPromise,
      ...opts
    });
  }
  async isExistingAlert(params) {
    return (0, _is_existing_alert.isExistingAlert)({
      logger: this.options.logger,
      esClient: await this.options.elasticsearchClientPromise,
      ...params
    });
  }
  async _updateMuteState({
    muted,
    targets,
    indices,
    logger
  }) {
    if (!indices || indices.length === 0) {
      throw new Error(`Unable to update mute state for rules (example: ${JSON.stringify(targets[0])}) - no alert indices available`);
    }
    if (targets.length === 0) {
      return;
    }
    const esClient = await this.options.elasticsearchClientPromise;
    const shouldClauses = targets.map(target => {
      const must = [{
        term: {
          [_ruleDataUtils.ALERT_RULE_UUID]: target.ruleId
        }
      }];
      if (target.alertInstanceIds) {
        must.push({
          terms: {
            [_ruleDataUtils.ALERT_INSTANCE_ID]: target.alertInstanceIds
          }
        });
      }
      return {
        bool: {
          must
        }
      };
    });
    const query = {
      bool: {
        must: [{
          term: {
            [_ruleDataUtils.ALERT_STATUS]: _ruleDataUtils.ALERT_STATUS_ACTIVE
          }
        }],
        should: shouldClauses,
        minimum_should_match: 1
      }
    };
    try {
      await esClient.updateByQuery({
        index: indices,
        conflicts: 'proceed',
        wait_for_completion: false,
        refresh: true,
        ignore_unavailable: true,
        query,
        script: {
          source: `ctx._source['${_ruleDataUtils.ALERT_MUTED}'] = params.muted;`,
          lang: 'painless',
          params: {
            muted
          }
        }
      });
    } catch (error) {
      logger.error(`Error updating muted field to ${muted} for ${targets.length} targets (example: ${JSON.stringify(targets[0])}) - ${error.message}`);
      throw error;
    }
  }
  async muteAlertInstance({
    ruleId,
    alertInstanceId,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: true,
      targets: [{
        ruleId,
        alertInstanceIds: [alertInstanceId]
      }],
      indices,
      logger
    });
  }
  async unmuteAlertInstance({
    ruleId,
    alertInstanceId,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: false,
      targets: [{
        ruleId,
        alertInstanceIds: [alertInstanceId]
      }],
      indices,
      logger
    });
  }
  async muteAllAlerts({
    ruleId,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: true,
      targets: [{
        ruleId
      }],
      indices,
      logger
    });
  }
  async unmuteAllAlerts({
    ruleId,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: false,
      targets: [{
        ruleId
      }],
      indices,
      logger
    });
  }
  async muteAlertInstances({
    targets,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: true,
      targets,
      indices,
      logger
    });
  }
  async unmuteAlertInstances({
    targets,
    indices,
    logger
  }) {
    return this._updateMuteState({
      muted: false,
      targets,
      indices,
      logger
    });
  }
}
exports.AlertsService = AlertsService;