"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.listIndicesTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const listIndicesSchema = _zod.z.object({
  pattern: _zod.z.string().default('*').describe(`Index pattern to match Elasticsearch indices, aliases and datastream names.
      - Correct examples: '.logs-*', '*data*', 'metrics-prod-*', 'my-specific-index', '*'
      - Should only be used if you are certain of a specific index pattern to filter on. *Do not try to guess*.
      - Defaults to '*' to match all indices.`)
});
const listIndicesTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.listIndices,
    type: _onechatCommon.ToolType.builtin,
    description: `List the indices, aliases and datastreams from the Elasticsearch cluster.

The 'pattern' optional parameter is an index pattern which can be used to filter resources.
This parameter should only be used when you already know of a specific pattern to filter on,
e.g. if the user provided one. Otherwise, do not try to invent or guess a pattern.`,
    schema: listIndicesSchema,
    handler: async ({
      pattern
    }, {
      esClient,
      logger
    }) => {
      logger.debug(`list indices tool called with pattern: ${pattern}`);
      const {
        indices,
        data_streams: dataStreams,
        aliases,
        warnings
      } = await (0, _onechatGenaiUtils.listSearchSources)({
        pattern,
        includeHidden: false,
        includeKibanaIndices: false,
        excludeIndicesRepresentedAsAlias: false,
        excludeIndicesRepresentedAsDatastream: true,
        esClient: esClient.asCurrentUser
      });
      return {
        results: [{
          type: _tool_result.ToolResultType.other,
          data: {
            indices: indices.map(index => ({
              name: index.name
            })),
            aliases: aliases.map(alias => ({
              name: alias.name,
              indices: alias.indices
            })),
            data_streams: dataStreams.map(ds => ({
              name: ds.name,
              indices: ds.indices
            })),
            warnings
          }
        }]
      };
    },
    tags: []
  };
};
exports.listIndicesTool = listIndicesTool;