"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flyoutManagerReducer = flyoutManagerReducer;
exports.initialState = void 0;
var _actions = require("./actions");
var _const = require("./const");
function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
function _toConsumableArray(r) { return _arrayWithoutHoles(r) || _iterableToArray(r) || _unsupportedIterableToArray(r) || _nonIterableSpread(); }
function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _iterableToArray(r) { if ("undefined" != typeof Symbol && null != r[Symbol.iterator] || null != r["@@iterator"]) return Array.from(r); }
function _arrayWithoutHoles(r) { if (Array.isArray(r)) return _arrayLikeToArray(r); }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == _typeof(i) ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != _typeof(t) || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != _typeof(i)) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
/**
 * Default flyout manager state used to initialize the reducer.
 */
var initialState = exports.initialState = {
  sessions: [],
  flyouts: [],
  layoutMode: _const.LAYOUT_MODE_SIDE_BY_SIDE,
  pushPadding: {
    left: 0,
    right: 0
  },
  currentZIndex: 0,
  unmanagedFlyouts: []
};

/**
 * Reducer handling all flyout manager actions and state transitions.
 */
function flyoutManagerReducer() {
  var state = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : initialState;
  var action = arguments.length > 1 ? arguments[1] : undefined;
  switch (action.type) {
    case _actions.ACTION_ADD_UNMANAGED_FLYOUT:
      {
        if (state.unmanagedFlyouts.includes(action.flyoutId)) {
          return state;
        }
        return _objectSpread(_objectSpread({}, state), {}, {
          // Increment by 2 for each new unmanaged flyout.
          // Unmanaged flyouts render on z-index of `n`, and their overlay mask
          // on `n - 1`.
          currentZIndex: state.currentZIndex + 2,
          unmanagedFlyouts: [].concat(_toConsumableArray(state.unmanagedFlyouts), [action.flyoutId])
        });
      }
    case _actions.ACTION_CLOSE_UNMANAGED_FLYOUT:
      {
        var newUnmanagedFlyouts = state.unmanagedFlyouts.filter(function (flyoutId) {
          return flyoutId !== action.flyoutId;
        });
        var newCurrentZIndex = state.currentZIndex;
        if (state.sessions.length === 0 && newUnmanagedFlyouts.length === 0) {
          newCurrentZIndex = 0;
        }
        return _objectSpread(_objectSpread({}, state), {}, {
          unmanagedFlyouts: newUnmanagedFlyouts,
          currentZIndex: newCurrentZIndex
        });
      }

    // Register a flyout.
    // - Ignore duplicates by `flyoutId`.
    // - For a `main` flyout, start a new session { main, child: null }.
    // - For a `child` flyout, attach it to the most recent session; if no
    //   session exists, do nothing (invalid child without a parent).
    case _actions.ACTION_ADD:
      {
        var flyoutId = action.flyoutId,
          title = action.title,
          level = action.level,
          size = action.size;

        // Ignore duplicate registrations
        if (state.flyouts.some(function (f) {
          return f.flyoutId === flyoutId;
        })) {
          return state;
        }
        var newFlyoutState = {
          level: level,
          flyoutId: flyoutId,
          size: size,
          activityStage: _const.STAGE_OPENING
        };
        var newFlyouts = [].concat(_toConsumableArray(state.flyouts), [newFlyoutState]);
        if (level === _const.LEVEL_MAIN) {
          var newSession = {
            mainFlyoutId: flyoutId,
            title: title,
            childFlyoutId: null,
            zIndex: state.currentZIndex
          };
          return _objectSpread(_objectSpread({}, state), {}, {
            sessions: [].concat(_toConsumableArray(state.sessions), [newSession]),
            flyouts: newFlyouts,
            // Increment by 3 for each new flyout session.
            // Managed flyouts render main flyouts on z-index of `n`,
            // child flyouts on `n - 1` and the overlay mask on `n - 2`.
            currentZIndex: state.currentZIndex + 3
          });
        }
        if (state.sessions.length === 0) {
          return state;
        }
        var updatedSessions = _toConsumableArray(state.sessions);
        var currentSessionIndex = updatedSessions.length - 1;
        updatedSessions[currentSessionIndex] = _objectSpread(_objectSpread({}, updatedSessions[currentSessionIndex]), {}, {
          childFlyoutId: flyoutId
        });
        return _objectSpread(_objectSpread({}, state), {}, {
          sessions: updatedSessions,
          flyouts: newFlyouts
        });
      }

    // Unregister a flyout and update sessions accordingly.
    // - When closing a `main` flyout, drop its entire session and all associated flyouts.
    // - When closing a `child` flyout, clear the child pointer on the most
    //   recent session if it matches.
    case _actions.ACTION_CLOSE:
      {
        var removedFlyout = state.flyouts.find(function (f) {
          return f.flyoutId === action.flyoutId;
        });
        if (!removedFlyout) {
          return state;
        }
        if (removedFlyout.level === _const.LEVEL_MAIN) {
          // Find the session that contains this main flyout
          var sessionToRemove = state.sessions.find(function (session) {
            return session.mainFlyoutId === action.flyoutId;
          });
          if (sessionToRemove) {
            // Remove all flyouts associated with this session (main + child)
            var flyoutsToRemove = new Set([action.flyoutId]);
            if (sessionToRemove.childFlyoutId) {
              flyoutsToRemove.add(sessionToRemove.childFlyoutId);
            }
            var _newFlyouts = state.flyouts.filter(function (f) {
              return !flyoutsToRemove.has(f.flyoutId);
            });
            var newSessions = state.sessions.filter(function (session) {
              return session.mainFlyoutId !== action.flyoutId;
            });
            var _newCurrentZIndex = state.currentZIndex;
            if (newSessions.length === 0 && state.unmanagedFlyouts.length === 0) {
              // Reset to initial value if no flyouts remain open to avoid
              // the value going too high during the lifecycle of the app
              _newCurrentZIndex = 0;
            }
            return _objectSpread(_objectSpread({}, state), {}, {
              sessions: newSessions,
              flyouts: _newFlyouts,
              currentZIndex: _newCurrentZIndex
            });
          }
        }

        // Handle child flyout closing (existing logic)
        var _newFlyouts2 = state.flyouts.filter(function (f) {
          return f.flyoutId !== action.flyoutId;
        });
        if (state.sessions.length === 0) {
          return _objectSpread(_objectSpread({}, state), {}, {
            flyouts: _newFlyouts2
          });
        }
        var _updatedSessions = _toConsumableArray(state.sessions);
        var _currentSessionIndex = _updatedSessions.length - 1;
        if (_updatedSessions[_currentSessionIndex].childFlyoutId === action.flyoutId) {
          _updatedSessions[_currentSessionIndex] = _objectSpread(_objectSpread({}, _updatedSessions[_currentSessionIndex]), {}, {
            childFlyoutId: null
          });
        }
        return _objectSpread(_objectSpread({}, state), {}, {
          sessions: _updatedSessions,
          flyouts: _newFlyouts2
        });
      }

    // Mark the provided flyout ID as the active child for the latest session.
    case _actions.ACTION_SET_ACTIVE:
      {
        // No-op when no session exists.
        if (state.sessions.length === 0) {
          return state;
        }
        var _updatedSessions2 = _toConsumableArray(state.sessions);
        var _currentSessionIndex2 = _updatedSessions2.length - 1;
        _updatedSessions2[_currentSessionIndex2] = _objectSpread(_objectSpread({}, _updatedSessions2[_currentSessionIndex2]), {}, {
          childFlyoutId: action.flyoutId
        });
        return _objectSpread(_objectSpread({}, state), {}, {
          sessions: _updatedSessions2
        });
      }

    // Persist a flyout's measured width (px). Used for responsive layout
    // calculations, e.g., deciding stacked vs side-by-side.
    case _actions.ACTION_SET_WIDTH:
      {
        var _flyoutId = action.flyoutId,
          width = action.width;
        var updatedFlyouts = state.flyouts.map(function (flyout) {
          return flyout.flyoutId === _flyoutId ? _objectSpread(_objectSpread({}, flyout), {}, {
            width: width
          }) : flyout;
        });
        return _objectSpread(_objectSpread({}, state), {}, {
          flyouts: updatedFlyouts
        });
      }

    // Switch global layout mode for managed flyouts.
    case _actions.ACTION_SET_LAYOUT_MODE:
      {
        return _objectSpread(_objectSpread({}, state), {}, {
          layoutMode: action.layoutMode
        });
      }

    // Update a flyout's activity stage in state
    case _actions.ACTION_SET_ACTIVITY_STAGE:
      {
        var _flyoutId2 = action.flyoutId;
        var _updatedFlyouts = state.flyouts.map(function (flyout) {
          return flyout.flyoutId === _flyoutId2 ? _objectSpread(_objectSpread({}, flyout), {}, {
            activityStage: action.activityStage
          }) : flyout;
        });
        return _objectSpread(_objectSpread({}, state), {}, {
          flyouts: _updatedFlyouts
        });
      }

    // Go back one session (remove current session from stack)
    case _actions.ACTION_GO_BACK:
      {
        if (state.sessions.length === 0) {
          return state;
        }
        var _currentSessionIndex3 = state.sessions.length - 1;
        var currentSession = state.sessions[_currentSessionIndex3];

        // Close all flyouts in the current session
        var _flyoutsToRemove = new Set([currentSession.mainFlyoutId]);
        if (currentSession.childFlyoutId) {
          _flyoutsToRemove.add(currentSession.childFlyoutId);
        }
        var _newFlyouts3 = state.flyouts.filter(function (f) {
          return !_flyoutsToRemove.has(f.flyoutId);
        });
        var _newSessions = state.sessions.slice(0, _currentSessionIndex3);
        return _objectSpread(_objectSpread({}, state), {}, {
          sessions: _newSessions,
          flyouts: _newFlyouts3
        });
      }

    // Navigate to a specific flyout (remove all sessions after it)
    case _actions.ACTION_GO_TO_FLYOUT:
      {
        var _flyoutId3 = action.flyoutId;

        // Find the session containing the target flyout
        var targetSessionIndex = state.sessions.findIndex(function (session) {
          return session.mainFlyoutId === _flyoutId3;
        });
        if (targetSessionIndex === -1) {
          return state; // Target flyout not found
        }

        // Close all sessions after the target session
        var sessionsToClose = state.sessions.slice(targetSessionIndex + 1);
        var _flyoutsToRemove2 = new Set();
        sessionsToClose.forEach(function (session) {
          _flyoutsToRemove2.add(session.mainFlyoutId);
          if (session.childFlyoutId) {
            _flyoutsToRemove2.add(session.childFlyoutId);
          }
        });
        var _newFlyouts4 = state.flyouts.filter(function (f) {
          return !_flyoutsToRemove2.has(f.flyoutId);
        });
        var _newSessions2 = state.sessions.slice(0, targetSessionIndex + 1);
        return _objectSpread(_objectSpread({}, state), {}, {
          sessions: _newSessions2,
          flyouts: _newFlyouts4
        });
      }

    // Set push padding offset for a specific side
    case _actions.ACTION_SET_PUSH_PADDING:
      {
        var _state$pushPadding;
        var side = action.side,
          _width = action.width;
        return _objectSpread(_objectSpread({}, state), {}, {
          pushPadding: _objectSpread(_objectSpread({}, (_state$pushPadding = state.pushPadding) !== null && _state$pushPadding !== void 0 ? _state$pushPadding : {
            left: 0,
            right: 0
          }), {}, _defineProperty({}, side, _width))
        });
      }
    default:
      return state;
  }
}