"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRafDebouncedCallback = useRafDebouncedCallback;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Hook that returns a debounced callback using `requestAnimationFrame`.
 *
 * @param fn - the callback function to debounce.
 */
function useRafDebouncedCallback(fn) {
  const rafIdRef = (0, _react.useRef)(null);
  const fnRef = (0, _react.useRef)(fn);

  // Always call the latest fn
  (0, _react.useLayoutEffect)(() => {
    fnRef.current = fn;
  }, [fn]);
  const schedule = (0, _react.useCallback)(() => {
    if (rafIdRef.current != null) cancelAnimationFrame(rafIdRef.current);
    rafIdRef.current = requestAnimationFrame(() => {
      rafIdRef.current = null;
      fnRef.current();
    });
  }, []);
  const cancel = (0, _react.useCallback)(() => {
    if (rafIdRef.current != null) {
      cancelAnimationFrame(rafIdRef.current);
      rafIdRef.current = null;
    }
  }, []);

  // Auto-cancel on unmount
  (0, _react.useLayoutEffect)(() => () => cancel(), [cancel]);
  return [schedule, cancel];
}