"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNamespacesBoolFilter = getNamespacesBoolFilter;
exports.getQueryParams = getQueryParams;
var esKuery = _interopRequireWildcard(require("@kbn/es-query"));
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _references_filter = require("./references_filter");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Gets the types based on the type. Uses mappings to support
 * null type (all types), a single type string or an array
 */
function getTypes(registry, type) {
  if (!type) {
    return registry.getAllTypes().map(registeredType => registeredType.name);
  }
  return Array.isArray(type) ? type : [type];
}

/**
 *  Get the field params based on the types, searchFields, and rootSearchFields
 */
function getSimpleQueryStringTypeFields(types, searchFields = [], rootSearchFields = []) {
  if (!searchFields.length && !rootSearchFields.length) {
    return {
      lenient: true,
      fields: ['*']
    };
  }
  let fields = [...rootSearchFields];
  fields.forEach(field => {
    if (field.indexOf('.') !== -1) {
      throw new Error(`rootSearchFields entry "${field}" is invalid: cannot contain "." character`);
    }
  });
  for (const field of searchFields) {
    fields = fields.concat(types.map(prefix => `${prefix}.${field}`));
  }
  return {
    fields
  };
}

/**
 *  Gets the clause that will filter for the type in the namespace.
 *  Some types are namespace agnostic, so they must be treated differently.
 */
function getClauseForType(registry, namespaces = [_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING], type) {
  if (namespaces.length === 0) {
    throw new Error('cannot specify empty namespaces array');
  }
  const searchAcrossAllNamespaces = namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING);
  if (registry.isMultiNamespace(type)) {
    const typeFilterClause = {
      term: {
        type
      }
    };
    const namespacesFilterClause = {
      terms: {
        namespaces: [...namespaces, _coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING]
      }
    };
    const must = searchAcrossAllNamespaces ? [typeFilterClause] : [typeFilterClause, namespacesFilterClause];
    return {
      bool: {
        must,
        must_not: [{
          exists: {
            field: 'namespace'
          }
        }]
      }
    };
  } else if (registry.isSingleNamespace(type)) {
    const should = [];
    const eligibleNamespaces = namespaces.filter(x => x !== _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING);
    if (eligibleNamespaces.length > 0 && !searchAcrossAllNamespaces) {
      should.push({
        terms: {
          namespace: eligibleNamespaces
        }
      });
    }
    if (namespaces.includes(_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING)) {
      should.push({
        bool: {
          must_not: [{
            exists: {
              field: 'namespace'
            }
          }]
        }
      });
    }
    const shouldClauseProps = should.length > 0 ? {
      should,
      minimum_should_match: 1
    } : {};
    return {
      bool: {
        must: [{
          term: {
            type
          }
        }],
        ...shouldClauseProps,
        must_not: [{
          exists: {
            field: 'namespaces'
          }
        }]
      }
    };
  }
  // isNamespaceAgnostic
  return {
    bool: {
      must: [{
        term: {
          type
        }
      }],
      must_not: [{
        exists: {
          field: 'namespace'
        }
      }, {
        exists: {
          field: 'namespaces'
        }
      }]
    }
  };
}
// A de-duplicated set of namespaces makes for a more efficient query.
const uniqNamespaces = namespacesToNormalize => namespacesToNormalize ? Array.from(new Set(namespacesToNormalize)) : undefined;
const toArray = val => {
  if (typeof val === 'undefined') {
    return val;
  }
  return !Array.isArray(val) ? [val] : val;
};
function getNamespacesBoolFilter({
  namespaces,
  registry,
  types,
  typeToNamespacesMap
}) {
  return {
    bool: {
      should: types.map(shouldType => {
        const deduplicatedNamespaces = uniqNamespaces(typeToNamespacesMap ? typeToNamespacesMap.get(shouldType) : namespaces);
        return getClauseForType(registry, deduplicatedNamespaces, shouldType);
      }),
      minimum_should_match: 1
    }
  };
}
/**
 *  Get the "query" related keys for the search body
 */
function getQueryParams({
  registry,
  namespaces,
  type,
  typeToNamespacesMap,
  search,
  searchFields: searchFieldsParam = [],
  rootSearchFields,
  defaultSearchOperator,
  hasReference,
  hasReferenceOperator,
  hasNoReference,
  hasNoReferenceOperator,
  kueryNode,
  mappings
}) {
  var _hasReference, _hasNoReference;
  const types = getTypes(registry, typeToNamespacesMap ? Array.from(typeToNamespacesMap.keys()) : type);
  hasReference = toArray(hasReference);
  hasNoReference = toArray(hasNoReference);
  const bool = {
    filter: [...(kueryNode != null ? [esKuery.toElasticsearchQuery(kueryNode)] : []), ...((_hasReference = hasReference) !== null && _hasReference !== void 0 && _hasReference.length ? [(0, _references_filter.getReferencesFilter)({
      references: hasReference,
      operator: hasReferenceOperator
    })] : []), ...((_hasNoReference = hasNoReference) !== null && _hasNoReference !== void 0 && _hasNoReference.length ? [(0, _references_filter.getReferencesFilter)({
      references: hasNoReference,
      operator: hasNoReferenceOperator,
      must: false
    })] : []), getNamespacesBoolFilter({
      namespaces,
      registry,
      types,
      typeToNamespacesMap
    })]
  };
  if (search) {
    const useMatchPhrasePrefix = shouldUseMatchPhrasePrefix(search);
    const isSearchingInAllFields = searchFieldsParam.length === 0 || searchFieldsParam.length === 1 && searchFieldsParam[0] === '*';
    const {
      simpleQuerySearchFields,
      nestedQueryFields
    } = getFieldsByQueryType({
      searchFields: searchFieldsParam,
      types,
      mappings
    });
    const useNestedStringClause = nestedQueryFields.size > 0;
    const useSimpleQueryStringClause = isSearchingInAllFields || simpleQuerySearchFields.length > 0;
    const useSimpleQueryStringClauseOnly = !useNestedStringClause && !useMatchPhrasePrefix;
    const useNestedStringClauseOnly = !useSimpleQueryStringClause && !useMatchPhrasePrefix;
    const simpleQueryStringClause = useSimpleQueryStringClause ? getSimpleQueryStringClause({
      search,
      searchFields: simpleQuerySearchFields,
      types,
      rootSearchFields,
      defaultSearchOperator
    }) : [];
    const nestedStringClause = useNestedStringClause ? getNestedQueryStringClause({
      search,
      nestedFields: nestedQueryFields
    }) : [];
    const matchPhrasePrefixClause = useMatchPhrasePrefix ? getMatchPhrasePrefixClauses({
      search,
      searchFields: simpleQuerySearchFields,
      types,
      registry
    }) : [];
    if (useSimpleQueryStringClauseOnly) {
      bool.must = [...simpleQueryStringClause];
    } else if (useNestedStringClauseOnly) {
      bool.must = [...nestedStringClause];
    } else {
      bool.should = [...simpleQueryStringClause, ...nestedStringClause, ...matchPhrasePrefixClause].filter(Boolean);
      bool.minimum_should_match = 1;
    }
  }
  return {
    query: {
      bool
    }
  };
}

// we only want to add match_phrase_prefix clauses
// if the search is a prefix search
const shouldUseMatchPhrasePrefix = search => {
  return search.trim().endsWith('*');
};
const getMatchPhrasePrefixClauses = ({
  search,
  searchFields,
  registry,
  types
}) => {
  // need to remove the prefix search operator
  const query = search.replace(/[*]$/, '');
  const mppFields = getMatchPhrasePrefixFields({
    searchFields,
    types,
    registry
  });
  return mppFields.map(({
    field,
    boost
  }) => {
    return {
      match_phrase_prefix: {
        [field]: {
          query,
          boost
        }
      }
    };
  });
};
const getMatchPhrasePrefixFields = ({
  searchFields = [],
  types,
  registry
}) => {
  const output = [];
  searchFields = searchFields.filter(field => field !== '*');
  let fields;
  if (searchFields.length === 0) {
    fields = types.reduce((typeFields, type) => {
      var _registry$getType, _registry$getType$man;
      const defaultSearchField = (_registry$getType = registry.getType(type)) === null || _registry$getType === void 0 ? void 0 : (_registry$getType$man = _registry$getType.management) === null || _registry$getType$man === void 0 ? void 0 : _registry$getType$man.defaultSearchField;
      if (defaultSearchField) {
        typeFields.push(`${type}.${defaultSearchField}`);
      }
      return typeFields;
    }, []);
  } else {
    fields = [];
    for (const field of searchFields) {
      fields = fields.concat(types.map(type => `${type}.${field}`));
    }
  }
  fields.forEach(rawField => {
    const [field, rawBoost] = rawField.split('^');
    let boost = 1;
    if (rawBoost) {
      try {
        boost = parseInt(rawBoost, 10);
      } catch (e) {
        boost = 1;
      }
    }
    if (isNaN(boost)) {
      boost = 1;
    }
    output.push({
      field,
      boost
    });
  });
  return output;
};
const getFieldsByQueryType = ({
  searchFields,
  types,
  mappings
}) => {
  const simpleQuerySearchFields = new Set();
  const nestedQueryFields = new Map();
  types.forEach(type => {
    searchFields.forEach(searchField => {
      var _getProperty;
      const isFieldDefinedAsNested = searchField.split('.').length > 1;
      const absoluteFieldPath = `${type}.${searchField}`;
      const parentNode = absoluteFieldPath.split('.').slice(0, -1).join('.');
      const parentNodeType = (_getProperty = (0, _coreSavedObjectsBaseServerInternal.getProperty)(mappings, parentNode)) === null || _getProperty === void 0 ? void 0 : _getProperty.type;
      if (isFieldDefinedAsNested && parentNodeType === 'nested') {
        nestedQueryFields.set(parentNode, [...(nestedQueryFields.get(parentNode) || []), `${type}.${searchField}`]);
      } else {
        simpleQuerySearchFields.add(searchField);
      }
    });
  });
  return {
    simpleQuerySearchFields: Array.from(simpleQuerySearchFields.values()),
    nestedQueryFields
  };
};

/**
 * Returns an array of clauses because there for each type we need to create a nested query
 */
const getNestedQueryStringClause = ({
  search,
  nestedFields
}) => {
  if (nestedFields.size === 0) {
    return [];
  }
  return Array.from(nestedFields.entries()).map(([path, fields]) => {
    return {
      nested: {
        path,
        query: {
          simple_query_string: {
            query: search,
            fields
          }
        }
      }
    };
  });
};
const getSimpleQueryStringClause = ({
  search,
  types,
  searchFields,
  rootSearchFields,
  defaultSearchOperator
}) => {
  return [{
    simple_query_string: {
      query: search,
      ...getSimpleQueryStringTypeFields(types, searchFields, rootSearchFields),
      ...(defaultSearchOperator ? {
        default_operator: defaultSearchOperator
      } : {})
    }
  }];
};