"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BraveSearchConnector = void 0;
var _v = require("@kbn/zod/v4");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Brave Search Connector
 *
 * Provides web search capabilities using the Brave Search API.
 * Features:
 * - Web search with customizable parameters
 * - Support for localization and safe search
 */

const DEFAULT_COUNT = 10;
const DEFAULT_OFFSET = 0;
const BraveSearchConnector = exports.BraveSearchConnector = {
  metadata: {
    id: '.brave-search',
    displayName: 'Brave Search',
    description: _i18n.i18n.translate('connectorSpecs.braveSearch.metadata.description', {
      defaultMessage: 'Search the web using Brave Search API for privacy-focused results'
    }),
    minimumLicense: 'enterprise',
    supportedFeatureIds: ['workflows']
  },
  auth: {
    types: [{
      type: 'api_key_header',
      defaults: {
        headerField: 'X-Subscription-Token'
      }
    }]
  },
  actions: {
    webSearch: {
      isTool: true,
      input: _v.z.object({
        q: _v.z.string().describe('Search query'),
        count: _v.z.number().int().min(1).max(20).optional().default(DEFAULT_COUNT).describe('Number of results to return (max 20)'),
        offset: _v.z.number().int().min(0).optional().default(DEFAULT_OFFSET).describe('Result offset for pagination')
      }),
      output: _v.z.object({
        query: _v.z.any().describe('Original query information'),
        results: _v.z.array(_v.z.any()).describe('Array of search results'),
        type: _v.z.string().describe('Search type')
      }),
      handler: async (ctx, input) => {
        var _response$data$web;
        const typedInput = input;
        const params = {
          q: typedInput.q,
          count: typedInput.count || DEFAULT_COUNT,
          offset: typedInput.offset || DEFAULT_OFFSET
        };
        const response = await ctx.client.get('https://api.search.brave.com/res/v1/web/search', {
          params,
          headers: {
            Accept: 'application/json',
            'Accept-Encoding': 'gzip'
          }
        });
        return {
          query: response.data.query,
          results: ((_response$data$web = response.data.web) === null || _response$data$web === void 0 ? void 0 : _response$data$web.results) || [],
          type: response.data.type
        };
      }
    }
  },
  test: {
    handler: async ctx => {
      try {
        // Perform a simple test search
        await ctx.client.get('https://api.search.brave.com/res/v1/web/search', {
          params: {
            q: 'test',
            count: 1
          },
          headers: {
            Accept: 'application/json',
            'Accept-Encoding': 'gzip'
          }
        });
        return {
          ok: true,
          message: 'Successfully connected to Brave Search API'
        };
      } catch (error) {
        const errorMessage = error instanceof Error ? error.message : String(error);
        return {
          ok: false,
          message: `Failed to connect to Brave Search API: ${errorMessage}`
        };
      }
    },
    description: _i18n.i18n.translate('connectorSpecs.braveSearch.test.description', {
      defaultMessage: 'Verifies Brave Search API key and connection'
    })
  }
};