"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentService = void 0;
var _agents = require("../../agents");
var _sub_agents = require("../../agents/sub_agents");
var _tools = require("../../agents/tools");
var _prompts = require("../../agents/prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AgentService {
  constructor(esClient, samplesIndexService, model) {
    this.esClient = esClient;
    this.samplesIndexService = samplesIndexService;
    this.model = model;
  }

  /**
   * Invokes the deep research agent with samples fetched from the index.
   * Uses tool-based approach:
   * - Service creates tools with samples and esClient
   * - Agent can fetch samples on demand using fetch_log_samples tool
   * - Validator tool has access to all samples
   * - No samples in context unless agent explicitly requests them (saves tokens)
   *
   * @param integration_id - The integration ID
   * @param data_stream_id - The data stream ID
   */
  async invoke_deep_agent(integrationId, dataStreamId) {
    // Fetch samples from the index (decoupled from agent building)
    const samples = await this.samplesIndexService.getSamplesForDataStream(integrationId, dataStreamId, this.esClient);

    // Create tools at the service level
    // Tools capture samples and esClient in their closures
    const fetchSamplesToolInstance = (0, _tools.fetchSamplesTool)(samples);
    const validatorTool = (0, _tools.ingestPipelineValidatorTool)(this.esClient, samples);

    // Create the sub agents with tools
    const logsAnalyzerSubAgent = (0, _sub_agents.createLogsAnalyzerAgent)({
      prompt: `You have access to the fetch_log_samples tool. Use it to retrieve log samples, then analyze the format and provide structured analysis for ingest pipeline generation.
      <workflow>
        1. Call fetch_log_samples to retrieve 5-10 sample logs
        2. Analyze the samples to identify format, fields, and characteristics
        3. Provide structured analysis output as specified in your system prompt
      </workflow>`,
      tools: [fetchSamplesToolInstance]
    });
    const pipelineGeneratorSubAgent = (0, _sub_agents.createIngestPipelineGeneratorAgent)({
      name: 'ingest_pipeline_generator',
      description: 'Generates an Elasticsearch ingest pipeline for the provided log samples and documentation.',
      prompt: _prompts.INGEST_PIPELINE_GENERATOR_PROMPT,
      tools: [validatorTool],
      sampleCount: samples.length
    });

    // Create and invoke the agent
    const automaticImportAgent = (0, _agents.createAutomaticImportAgent)({
      model: this.model,
      subagents: [logsAnalyzerSubAgent, pipelineGeneratorSubAgent, _sub_agents.textToEcsSubAgent]
    });
    const result = await automaticImportAgent.invoke({
      messages: [{
        role: 'user',
        content: `You are tasked with generating an Elasticsearch ingest pipeline for the integration \`${integrationId}\` and data stream \`${dataStreamId}\`.`
      }]
    });
    return result;
  }
}
exports.AgentService = AgentService;