"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataStreamRoutes = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _utils = require("./utils");
var _feature = require("../feature");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerDataStreamRoutes = (router, logger) => {
  uploadSamplesRoute(router, logger);
  deleteDataStreamRoute(router, logger);
};
exports.registerDataStreamRoutes = registerDataStreamRoutes;
const uploadSamplesRoute = (router, logger) => router.versioned.post({
  access: 'internal',
  path: '/api/automatic_import_v2/integrations/{integration_id}/data_streams/{data_stream_id}/upload',
  security: {
    authz: {
      requiredPrivileges: [`${_feature.AUTOMATIC_IMPORT_API_PRIVILEGES.MANAGE}`]
    }
  }
}).addVersion({
  version: '1',
  validate: {
    request: {
      params: (0, _zodHelpers.buildRouteValidationWithZod)(_common.UploadSamplesToDataStreamRequestParams),
      body: (0, _zodHelpers.buildRouteValidationWithZod)(_common.UploadSamplesToDataStreamRequestBody)
    }
  }
}, async (context, request, response) => {
  try {
    const automaticImportv2 = await context.automaticImportv2;
    const automaticImportService = automaticImportv2.automaticImportService;
    const currentUser = await automaticImportv2.getCurrentUser();
    const esClient = automaticImportv2.esClient;
    const {
      integration_id: integrationId,
      data_stream_id: dataStreamId
    } = request.params;
    const {
      samples,
      originalSource
    } = request.body;
    const result = await automaticImportService.addSamplesToDataStream({
      integrationId,
      dataStreamId,
      rawSamples: samples,
      originalSource,
      authenticatedUser: currentUser,
      esClient
    });
    return response.ok({
      body: result
    });
  } catch (err) {
    logger.error(`registerDataStreamRoutes: Caught error:`, err);
    const automaticImportResponse = (0, _utils.buildAutomaticImportResponse)(response);
    return automaticImportResponse.error({
      statusCode: 500,
      body: err
    });
  }
});
const deleteDataStreamRoute = (router, logger) => router.versioned.delete({
  access: 'internal',
  path: '/api/automatic_import_v2/integrations/{integration_id}/data_streams/{data_stream_id}',
  security: {
    authz: {
      requiredPrivileges: [`${_feature.AUTOMATIC_IMPORT_API_PRIVILEGES.MANAGE}`]
    }
  }
}).addVersion({
  version: '1',
  validate: {
    request: {
      params: (0, _zodHelpers.buildRouteValidationWithZod)(_common.DeleteDataStreamRequestParams)
    }
  }
}, async (context, request, response) => {
  try {
    const automaticImportv2 = await context.automaticImportv2;
    const automaticImportService = automaticImportv2.automaticImportService;
    const {
      integration_id: integrationId,
      data_stream_id: dataStreamId
    } = request.params;
    const esClient = automaticImportv2.esClient;
    await automaticImportService.deleteDataStream(integrationId, dataStreamId, esClient);
    return response.ok();
  } catch (err) {
    logger.error(`deleteDataStreamRoute: Caught error:`, err);
    const automaticImportResponse = (0, _utils.buildAutomaticImportResponse)(response);
    return automaticImportResponse.error({
      statusCode: 500,
      body: err
    });
  }
});