"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildPackage = buildPackage;
exports.isValidDatastreamName = isValidDatastreamName;
exports.isValidName = isValidName;
exports.renderPackageManifestYAML = renderPackageManifestYAML;
var _utils = require("@kbn/utils");
var _admZip = _interopRequireDefault(require("adm-zip"));
var _jsYaml = require("js-yaml");
var _nunjucks = _interopRequireDefault(require("nunjucks"));
var _path = require("path");
var _constants = require("../../common/constants");
var _util = require("../util");
var _samples = require("../util/samples");
var _agent = require("./agent");
var _constants2 = require("./constants");
var _data_stream = require("./data_stream");
var _fields = require("./fields");
var _pipeline = require("./pipeline");
var _readme_files = require("./readme_files");
var _build_integration_error = require("../lib/errors/build_integration_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initialVersion = '1.0.0';
function configureNunjucks() {
  const templateDir = (0, _path.join)(__dirname, '../templates');
  const agentTemplates = (0, _path.join)(templateDir, 'agent');
  const manifestTemplates = (0, _path.join)(templateDir, 'manifest');
  const systemTestTemplates = (0, _path.join)(templateDir, 'system_tests');
  _nunjucks.default.configure([templateDir, agentTemplates, manifestTemplates, systemTestTemplates], {
    autoescape: false
  });
}
async function buildPackage(integration) {
  configureNunjucks();
  if (!isValidName(integration.name)) {
    throw new _build_integration_error.BuildIntegrationError(`Invalid integration name: ${integration.name}, Should only contain letters, numbers and underscores`);
  }
  const workingDir = (0, _path.join)((0, _utils.getDataPath)(), `automatic-import-${(0, _util.generateUniqueId)()}`);
  try {
    const packageDirectoryName = `${integration.name}-${initialVersion}`;
    const packageDir = createDirectories(workingDir, integration, packageDirectoryName);
    const dataStreamsDir = (0, _path.join)(packageDir, 'data_stream');
    const fieldsPerDatastream = integration.dataStreams.map(dataStream => {
      const dataStreamName = dataStream.name;
      if (!isValidDatastreamName(dataStreamName)) {
        throw new Error(`Invalid datastream name: ${dataStreamName}, Name must be at least 2 characters long and can only contain lowercase letters, numbers, and underscores`);
      }
      const specificDataStreamDir = (0, _path.join)(dataStreamsDir, dataStreamName);
      const dataStreamFields = (0, _data_stream.createDataStream)(integration.name, specificDataStreamDir, dataStream);
      (0, _agent.createAgentInput)(specificDataStreamDir, dataStream.inputTypes, dataStream.celInput);
      (0, _pipeline.createPipeline)(specificDataStreamDir, dataStream.pipeline);
      const fields = (0, _fields.createFieldMapping)(integration.name, dataStreamName, specificDataStreamDir, dataStream.docs);
      return {
        datastream: dataStreamName,
        fields: mergeAndSortFields(fields, dataStreamFields)
      };
    });
    (0, _readme_files.createReadme)(packageDir, integration.name, integration.dataStreams, fieldsPerDatastream);
    const zipBuffer = await createZipArchive(integration, workingDir, packageDirectoryName);
    (0, _util.removeDirSync)(workingDir);
    return zipBuffer;
  } catch (error) {
    throw new _build_integration_error.BuildIntegrationError('Building the Integration failed');
  } finally {
    (0, _util.removeDirSync)(workingDir);
  }
}
function isValidName(input) {
  return input.length > 0 && _constants.NAME_REGEX_PATTERN.test(input);
}
function isValidDatastreamName(input) {
  return input.length > 0 && _constants.DATASTREAM_NAME_REGEX_PATTERN.test(input);
}
function createDirectories(workingDir, integration, packageDirectoryName) {
  const packageDir = (0, _path.join)(workingDir, packageDirectoryName);
  (0, _util.ensureDirSync)(workingDir);
  (0, _util.ensureDirSync)(packageDir);
  createPackage(packageDir, integration);
  return packageDir;
}
function createPackage(packageDir, integration) {
  createChangelog(packageDir);
  createBuildFile(packageDir);
  createPackageManifest(packageDir, integration);
  //  Skipping creation of system tests temporarily for custom package generation
  //  createPackageSystemTests(packageDir, integration);
}
function createBuildFile(packageDir) {
  const buildFile = _nunjucks.default.render('build.yml.njk', {
    ecs_version: '8.11.0'
  });
  const buildDir = (0, _path.join)(packageDir, '_dev/build');
  (0, _util.ensureDirSync)(buildDir);
  (0, _util.createSync)((0, _path.join)(buildDir, 'build.yml'), buildFile);
}
function createChangelog(packageDir) {
  const changelogTemplate = _nunjucks.default.render('changelog.yml.njk', {
    initial_version: initialVersion
  });
  (0, _util.createSync)((0, _path.join)(packageDir, 'changelog.yml'), changelogTemplate);
}
async function createZipArchive(integration, workingDir, packageDirectoryName) {
  const tmpPackageDir = (0, _path.join)(workingDir, packageDirectoryName);
  const zip = new _admZip.default();
  zip.addLocalFolder(tmpPackageDir, packageDirectoryName);
  if (integration.logo) {
    const logoDir = (0, _path.join)(packageDirectoryName, 'img/logo.svg');
    const logoBuffer = Buffer.from(integration.logo, 'base64');
    zip.addFile(logoDir, logoBuffer);
  }
  const buffer = zip.toBuffer();
  return buffer;
}
function mergeAndSortFields(fields, dataStreamFields) {
  const mergedFields = [...fields, ...dataStreamFields];
  return (0, _samples.flattenObjectsList)(mergedFields);
}

/**
 * Creates a package manifest dictionary.
 *
 * @param format_version - The format version of the package.
 * @param package_title - The title of the package.
 * @param package_name - The name of the package.
 * @param package_version - The version of the package.
 * @param package_description - The description of the package.
 * @param package_logo - The package logo file name, if present.
 * @param package_owner - The owner of the package.
 * @param min_version - The minimum version of Kibana required for the package.
 * @param inputs - An array of unique input objects containing type, title, and description.
 * @returns The package manifest dictionary.
 */
function createPackageManifestDict(format_version, package_title, package_name, package_version, package_description, package_logo, package_owner, min_version, inputs) {
  const data = {
    format_version,
    name: package_name,
    title: package_title,
    version: package_version,
    description: package_description,
    type: 'integration',
    categories: ['security', 'iam'],
    conditions: {
      kibana: {
        version: min_version
      }
    },
    policy_templates: [{
      name: package_name,
      title: package_title,
      description: package_description,
      inputs: inputs.map(input => ({
        type: input.type,
        title: `${input.title} : ${input.type}`,
        description: input.description
      }))
    }],
    owner: {
      github: package_owner,
      type: 'community'
    }
  };
  if (package_logo !== undefined && package_logo !== '') {
    data.icons = [{
      src: '/img/logo.svg',
      title: `${package_title} Logo`,
      size: '32x32',
      type: 'image/svg+xml'
    }];
  }
  return data;
}

/**
 * Render the package manifest for an integration.
 *
 * @param integration - The integration object.
 * @returns The package manifest YAML rendered into a string.
 */
function renderPackageManifestYAML(integration) {
  const uniqueInputs = {};
  integration.dataStreams.forEach(dataStream => {
    dataStream.inputTypes.forEach(inputType => {
      if (!uniqueInputs[inputType]) {
        uniqueInputs[inputType] = {
          type: inputType,
          title: dataStream.title,
          description: dataStream.description
        };
      }
    });
  });
  const uniqueInputsList = Object.values(uniqueInputs);
  const packageData = createPackageManifestDict(_constants2.FORMAT_VERSION,
  // format_version
  integration.title,
  // package_title
  integration.name,
  // package_name
  initialVersion,
  // package_version
  integration.description,
  // package_description
  integration.logo,
  // package_logo
  '@elastic/custom-integrations',
  // package_owner
  _constants2.KIBANA_MINIMUM_VERSION,
  // min_version
  uniqueInputsList // inputs
  );
  return (0, _jsYaml.dump)(packageData);
}
function createPackageManifest(packageDir, integration) {
  const packageManifest = renderPackageManifestYAML(integration);
  (0, _util.createSync)((0, _path.join)(packageDir, 'manifest.yml'), packageManifest);
}