"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGapAutoFillScheduler = createGapAutoFillScheduler;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _schemas = require("./schemas");
var _transform_gap_auto_fill_scheduler_param_to_saved_object = require("./transforms/transform_gap_auto_fill_scheduler_param_to_saved_object");
var _transforms = require("../../transforms");
var _saved_objects = require("../../../../../saved_objects");
var _authorization = require("../../../../../authorization");
var _audit_events = require("../../../../../rules_client/common/audit_events");
var _scheduler = require("../../../types/scheduler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createGapAutoFillScheduler(context, params) {
  try {
    _schemas.createGapAutoFillSchedulerSchema.validate(params);
  } catch (error) {
    const {
      request: _omitRequest,
      ...otherParams
    } = params;
    throw _boom.default.badRequest(`Error validating gap auto fill scheduler parameters - ${error.message}
Payload summary: ${JSON.stringify(otherParams, (key, value) => key === 'request' ? undefined : value)}`);
  }
  try {
    for (const ruleType of params.ruleTypes) {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType.type,
        consumer: ruleType.consumer,
        operation: _authorization.WriteOperations.CreateGapAutoFillScheduler,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    }
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.CREATE,
      error
    }));
    throw error;
  }
  try {
    var _context$auditLogger2;
    const soClient = context.unsecuredSavedObjectsClient;
    const taskManager = context.taskManager;
    const uniqueRuleTypeIds = new Set(params.ruleTypes.map(({
      type
    }) => type));

    // Throw error if a rule type is not registered
    for (const ruleTypeId of uniqueRuleTypeIds) {
      context.ruleTypeRegistry.get(ruleTypeId);
    }

    // Throw error if a gap auto fill scheduler already exists for the same (rule type, consumer) pair
    const pairs = Array.from(new Set(params.ruleTypes.map(rt => `${rt.type}:${rt.consumer}`)));
    if (pairs.length > 0) {
      const filter = `(${pairs.map(p => `${_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE}.attributes.ruleTypeConsumerPairs: "${p}"`).join(' or ')})`;
      const {
        saved_objects: candidates
      } = await soClient.find({
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        perPage: 1,
        filter
      });
      if (candidates.length > 0) {
        // Pairs are exact; any match is a duplicate
        throw _boom.default.conflict(`A gap auto fill scheduler already exists for at least one of the specified (rule type, consumer) pairs`);
      }
    }
    const createdBy = await context.getUserName();
    const now = new Date().toISOString();
    const attributes = (0, _transform_gap_auto_fill_scheduler_param_to_saved_object.transformGapAutoFillSchedulerCreateParamToSavedObject)(params, {
      createdBy,
      createdAt: now,
      updatedAt: now,
      updatedBy: createdBy
    });
    const savedObjectOptions = {
      refresh: 'wait_for'
    };
    if (params.id) {
      savedObjectOptions.id = params.id;
    }
    const so = await soClient.create(_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE, attributes, savedObjectOptions);
    if (params.enabled) {
      try {
        var _params$scope;
        await taskManager.ensureScheduled({
          id: so.id,
          taskType: _scheduler.GAP_AUTO_FILL_SCHEDULER_TASK_TYPE,
          schedule: params.schedule,
          scope: (_params$scope = params.scope) !== null && _params$scope !== void 0 ? _params$scope : [],
          params: {
            configId: so.id,
            spaceId: context.spaceId
          },
          state: {}
        }, {
          request: params.request
        });
      } catch (e) {
        context.logger.error(`Failed to schedule task for gap auto fill scheduler ${so.id}. Will attempt to delete the saved object.`);
        try {
          await soClient.delete(_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE, so.id);
        } catch (deleteError) {
          context.logger.error(`Failed to delete gap auto fill saved object for gap auto fill scheduler ${so.id}.`);
        }
        throw e;
      }
    }

    // Log successful creation
    (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.CREATE,
      savedObject: {
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        id: so.id,
        name: so.attributes.name
      }
    }));

    // Transform the saved object to the result format
    return (0, _transforms.transformSavedObjectToGapAutoFillSchedulerResult)({
      savedObject: so
    });
  } catch (error) {
    throw _boom.default.boomify(error, {
      message: 'Failed to create gap auto fill scheduler'
    });
  }
}