"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AIValueExportProvider = AIValueExportProvider;
exports.useAIValueExportContext = void 0;
var _react = _interopRequireWildcard(require("react"));
var _reactRouterDom = require("react-router-dom");
var _locator = require("../../../../common/locators/ai_value_report/locator");
var _kibana = require("../../../common/lib/kibana");
var _types = require("../../../common/lib/telemetry/events/ai_value_report/types");
var _jsxFileName = "/opt/buildkite-agent/builds/bk-agent-prod-gcp-1769602143372611309/elastic/kibana-artifacts-snapshot/kibana/x-pack/solutions/security/plugins/security_solution/public/reports/providers/ai_value/export_provider.tsx";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
const AIValueExportContext = /*#__PURE__*/(0, _react.createContext)(null);
const useAIValueExportContext = () => (0, _react.useContext)(AIValueExportContext);
exports.useAIValueExportContext = useAIValueExportContext;
const hashReportData = async data => {
  const str = JSON.stringify(data);
  const enc = new TextEncoder();
  const hash = await crypto.subtle.digest('SHA-256', enc.encode(str));
  return Array.from(new Uint8Array(hash)).map(v => v.toString(16).padStart(2, '0')).join('');
};

/**
 * This provider manages context for the AI Value Report.
 * It exposes hooks for setting the report’s input data and the AI-generated
 * cost-savings trend insight when the report is loaded.
 *
 * After these values are set, the `buildForwardedState` function becomes ready
 * to use when exporting the report to PDF. Only the AI-generated insight and a
 * hash of the input data it was derived from are included in the forwarded
 * state. This ensures the backend does not regenerate the insight (and consume
 * AI tokens) if the input data has not changed.
 *
 * If the navigation history contains a state (indicating that the page is being
 * exported, such as to PDF), the provider parses that state and extracts the
 * AI-generated insight along with the input-data hash. It then waits for the
 * report data to load, hashes it, and checks it against the forwarded hash.
 * If the hashes match, `shouldRegenerateInsight` is set to false; otherwise it
 * is set to true.
 */
function AIValueExportProvider({
  children
}) {
  const history = (0, _reactRouterDom.useHistory)();
  const [forwardedState, setForwardedState] = (0, _react.useState)();
  const [isInsightVerified, setIsInsightVerified] = (0, _react.useState)(false);
  const [shouldRegenerateInsight, setShouldRegenerateInsight] = (0, _react.useState)();
  const [reportInput, setReportInput] = (0, _react.useState)();
  const [reportDataHash, setReportDataHash] = (0, _react.useState)();
  const [insight, setInsight] = (0, _react.useState)();
  const abortControllerRef = (0, _react.useRef)(null);
  const [hashReportErrorMessage, setHashReportErrorMessage] = (0, _react.useState)('');
  const {
    telemetry
  } = (0, _kibana.useKibana)().services;
  const [isExportMode, setIsExportMode] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (history.location.state) {
      const parsedState = (0, _locator.parseLocationState)(history.location.state);

      // Only treat this as "export mode" when the forwarded state includes the data we need to
      // verify / reuse the pre-generated insight. Other navigations (e.g. "Open ..." from
      // Reporting) may include unrelated history state or only a timeRange, and should render the
      // normal UI (including the date picker).
      if (parsedState !== null && parsedState !== void 0 && parsedState.insight && parsedState !== null && parsedState !== void 0 && parsedState.reportDataHash) {
        setForwardedState(parsedState);
        setIsExportMode(true);
        return;
      }
    }
    setForwardedState(undefined);
    setIsExportMode(false);
  }, [history.location.state]);
  (0, _react.useEffect)(() => {
    if (reportInput) {
      setReportDataHash(undefined);
      if (abortControllerRef.current !== null) {
        abortControllerRef.current.abort();
      }
      const controller = new AbortController();
      abortControllerRef.current = controller;
      const generateReportDataHash = async () => {
        let hash;
        try {
          hash = await hashReportData(reportInput);
        } catch (e) {
          var _e$message;
          // Fallback to the date string which will force the regeneration of the insight
          hash = new Date().toISOString();
          setHashReportErrorMessage((_e$message = e === null || e === void 0 ? void 0 : e.message) !== null && _e$message !== void 0 ? _e$message : 'error during the hash generation');
        }
        if (controller.signal.aborted) {
          return;
        }
        setReportDataHash(hash);
      };
      generateReportDataHash();
    }
  }, [reportInput]);
  (0, _react.useEffect)(() => {
    if (forwardedState && reportDataHash) {
      setShouldRegenerateInsight(reportDataHash !== forwardedState.reportDataHash);
      setIsInsightVerified(true);
    }
  }, [forwardedState, reportDataHash]);

  // Telemetry reporting
  (0, _react.useEffect)(() => {
    if (isInsightVerified && shouldRegenerateInsight !== undefined) {
      telemetry.reportEvent(_types.AIValueReportEventTypes.AIValueReportExportInsightVerified, {
        shouldRegenerate: shouldRegenerateInsight
      });
    }
  }, [telemetry, isInsightVerified, shouldRegenerateInsight]);
  (0, _react.useEffect)(() => {
    if (isExportMode) {
      telemetry.reportEvent(_types.AIValueReportEventTypes.AIValueReportExportExecution, {});
    }
  }, [isExportMode, telemetry]);
  (0, _react.useEffect)(() => {
    if (hashReportErrorMessage) {
      telemetry.reportEvent(_types.AIValueReportEventTypes.AIValueReportExportError, {
        errorMessage: hashReportErrorMessage,
        isExportMode
      });
    }
  }, [hashReportErrorMessage, isExportMode, telemetry]);
  const buildForwardedState = (0, _react.useCallback)(({
    timeRange
  }) => {
    if (!insight || !reportDataHash) {
      return undefined;
    }
    return {
      timeRange,
      insight,
      reportDataHash
    };
  }, [insight, reportDataHash]);
  const value = (0, _react.useMemo)(() => ({
    isExportMode,
    forwardedState,
    isInsightVerified,
    shouldRegenerateInsight,
    buildForwardedState,
    setInsight,
    setReportInput
  }), [forwardedState, buildForwardedState, isInsightVerified, shouldRegenerateInsight, isExportMode]);
  return /*#__PURE__*/_react.default.createElement(AIValueExportContext.Provider, {
    value: value,
    __self: this,
    __source: {
      fileName: _jsxFileName,
      lineNumber: 193,
      columnNumber: 10
    }
  }, children);
}