"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.endpointMiddlewareFactory = void 0;
exports.handleLoadMetadataTransformStats = handleLoadMetadataTransformStats;
var _rxjs = require("rxjs");
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _authz = require("../../../../../common/endpoint/service/authz/authz");
var _constants = require("../../../../../common/endpoint/constants");
var _endpoint_isolation = require("../../../../common/lib/endpoint/endpoint_isolation");
var _policies = require("../../../services/policies/policies");
var _state = require("../../../state");
var _ingest = require("../../../services/policies/ingest");
var _selectors = require("./selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line no-console
const logError = console.error;
const endpointMiddlewareFactory = (coreStart, depsStart) => {
  // this needs to be called after endpointPackageVersion is loaded (getEndpointPackageInfo)
  // or else the wrong pattern might be loaded
  async function fetchIndexPatterns(state) {
    var _endpointPackageVersi;
    const packageVersion = (_endpointPackageVersi = (0, _selectors.endpointPackageVersion)(state)) !== null && _endpointPackageVersi !== void 0 ? _endpointPackageVersi : '';
    const parsedPackageVersion = packageVersion.includes('-') ? packageVersion.substring(0, packageVersion.indexOf('-')) : packageVersion;
    const minUnitedIndexVersion = '1.2.0';
    const indexPatternToFetch = (0, _gte.default)(parsedPackageVersion, minUnitedIndexVersion) ? _constants.METADATA_UNITED_INDEX : _constants.metadataCurrentIndexPattern;
    const res$ = depsStart.data.search.search({
      indices: [indexPatternToFetch],
      onlyCheckIfIndicesExist: false
    }, {
      strategy: _constants.ENDPOINT_FIELDS_SEARCH_STRATEGY
    });
    const response = await (0, _rxjs.firstValueFrom)(res$);
    const indexPattern = {
      title: indexPatternToFetch,
      fields: response.indexFields
    };
    return [indexPattern];
  }
  return store => next => async action => {
    next(action);
    const {
      getState,
      dispatch
    } = store;
    await getEndpointPackageInfo(getState(), dispatch, coreStart);

    // Endpoint list
    if ((action.type === 'userChangedUrl' || action.type === 'appRequestedEndpointList') && (0, _selectors.isOnEndpointPage)(getState())) {
      await endpointListMiddleware({
        coreStart,
        store,
        fetchIndexPatterns
      });
    }

    // Isolate Host
    if (action.type === 'endpointIsolationRequest') {
      return handleIsolateEndpointHost(store, action);
    }
    if (action.type === 'loadMetadataTransformStats') {
      return handleLoadMetadataTransformStats(coreStart.http, store);
    }
  };
};
exports.endpointMiddlewareFactory = endpointMiddlewareFactory;
const getNonExistingPoliciesForEndpointList = async (http, hosts, currentNonExistingPolicies) => {
  if (hosts.length === 0) {
    return;
  }

  // Create an array of unique policy IDs that are not yet known to be non-existing.
  const policyIdsToCheck = [...hosts.reduce((acc, host) => {
    const appliedPolicyId = host.metadata.Endpoint.policy.applied.id;
    if (!currentNonExistingPolicies.has(appliedPolicyId)) {
      acc.add(appliedPolicyId);
    }
    return acc;
  }, new Set())];
  if (policyIdsToCheck.length === 0) {
    return;
  }
  const policiesFound = (await (0, _ingest.sendBulkGetPackagePolicies)(http, policyIdsToCheck)).items.reduce((set, packagePolicy) => set.add(packagePolicy.id), new Set());
  const nonExistingPackagePolicies = policyIdsToCheck.reduce((set, policyId) => {
    if (!policiesFound.has(policyId)) {
      set.add(policyId);
    }
    return set;
  }, new Set());
  if (!nonExistingPackagePolicies.size) {
    return;
  }
  return nonExistingPackagePolicies;
};
const endpointsTotal = async http => {
  try {
    return (await http.get(_constants.HOST_METADATA_LIST_ROUTE, {
      version: '2023-10-31',
      query: {
        page: 0,
        pageSize: 1
      }
    })).total;
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    logError(`error while trying to check for total endpoints`);
    logError(error);
  }
  return 0;
};
const doEndpointsExist = async http => {
  try {
    return (await endpointsTotal(http)) > 0;
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    logError(`error while trying to check if endpoints exist`);
    logError(error);
  }
  return false;
};
const handleIsolateEndpointHost = async ({
  getState,
  dispatch
}, action) => {
  const state = getState();
  if ((0, _selectors.getIsIsolationRequestPending)(state)) {
    return;
  }
  dispatch({
    type: 'endpointIsolationRequestStateChange',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.getCurrentIsolationRequestState)(state)))
  });
  try {
    // Cast needed below due to the value of payload being `Immutable<>`
    let response;
    if (action.payload.type === 'unisolate') {
      response = await (0, _endpoint_isolation.unIsolateHost)(action.payload.data);
    } else {
      response = await (0, _endpoint_isolation.isolateHost)(action.payload.data);
    }
    dispatch({
      type: 'endpointIsolationRequestStateChange',
      payload: (0, _state.createLoadedResourceState)(response)
    });
  } catch (error) {
    var _error$body;
    dispatch({
      type: 'endpointIsolationRequestStateChange',
      payload: (0, _state.createFailedResourceState)((_error$body = error.body) !== null && _error$body !== void 0 ? _error$body : error)
    });
  }
};
async function getEndpointPackageInfo(state, dispatch, coreStart) {
  if (!(0, _selectors.getIsEndpointPackageInfoUninitialized)(state)) return;
  dispatch({
    type: 'endpointPackageInfoStateChanged',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.endpointPackageInfo)(state)))
  });
  try {
    const packageInfo = await (0, _ingest.sendGetEndpointSecurityPackage)(coreStart.http);
    dispatch({
      type: 'endpointPackageInfoStateChanged',
      payload: (0, _state.createLoadedResourceState)(packageInfo)
    });
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    // Ignore Errors, since this should not hinder the user's ability to use the UI
    logError(error);
    dispatch({
      type: 'endpointPackageInfoStateChanged',
      payload: (0, _state.createFailedResourceState)(error)
    });
  }
}
async function endpointListMiddleware({
  store,
  coreStart,
  fetchIndexPatterns
}) {
  const {
    getState,
    dispatch
  } = store;
  const {
    page_index: pageIndex,
    page_size: pageSize,
    sort_field: sortField,
    sort_direction: sortDirection
  } = (0, _selectors.uiQueryParams)(getState());
  let endpointResponse;
  try {
    const decodedQuery = (0, _selectors.searchBarQuery)(getState());
    endpointResponse = await coreStart.http.get(_constants.HOST_METADATA_LIST_ROUTE, {
      version: '2023-10-31',
      query: {
        page: pageIndex,
        pageSize,
        kuery: decodedQuery.query,
        sortField,
        sortDirection
      }
    });
    dispatch({
      type: 'serverReturnedEndpointList',
      payload: endpointResponse
    });
    fetchNonExistingPolicies({
      coreStart,
      hosts: endpointResponse.data,
      store
    });
  } catch (error) {
    dispatch({
      type: 'serverFailedToReturnEndpointList',
      payload: error
    });
  }

  // get an index pattern and fields for search bar
  if ((0, _selectors.patterns)(getState()).length === 0) {
    try {
      const indexPatterns = await fetchIndexPatterns(getState());
      if (indexPatterns !== undefined) {
        dispatch({
          type: 'serverReturnedMetadataPatterns',
          payload: indexPatterns
        });
      }
    } catch (error) {
      dispatch({
        type: 'serverFailedToReturnMetadataPatterns',
        payload: error
      });
    }
  }

  // No endpoints, so we should check to see if there are policies for onboarding
  if (endpointResponse && endpointResponse.data.length === 0) {
    const http = coreStart.http;

    // The original query to the list could have had an invalid param (ex. invalid page_size),
    // so we check first if endpoints actually do exist before pulling in data for the onboarding
    // messages.
    if (await doEndpointsExist(http)) {
      dispatch({
        type: 'serverFinishedInitialization',
        payload: true
      });
      return;
    }
    dispatch({
      type: 'serverReturnedEndpointExistValue',
      payload: false
    });
    if ((0, _authz.canFetchPackageAndAgentPolicies)(coreStart.application.capabilities)) {
      try {
        const policyDataResponse = await (0, _policies.sendGetEndpointSpecificPackagePolicies)(http, {
          query: {
            perPage: 50,
            // Since this is an onboarding flow, we'll cap at 50 policies.
            page: 1
          }
        });
        dispatch({
          type: 'serverReturnedPoliciesForOnboarding',
          payload: {
            policyItems: policyDataResponse.items
          }
        });
      } catch (error) {
        var _error$body2;
        dispatch({
          type: 'serverFailedToReturnPoliciesForOnboarding',
          payload: (_error$body2 = error.body) !== null && _error$body2 !== void 0 ? _error$body2 : error
        });
      }
    } else {
      dispatch({
        type: 'serverCancelledPolicyItemsLoading'
      });
    }
  } else {
    dispatch({
      type: 'serverCancelledPolicyItemsLoading'
    });
    dispatch({
      type: 'serverReturnedEndpointExistValue',
      payload: true
    });
  }
  dispatch({
    type: 'serverFinishedInitialization',
    payload: true
  });
}
async function handleLoadMetadataTransformStats(http, store) {
  const {
    getState,
    dispatch
  } = store;
  if (!http || !getState || !dispatch) {
    return;
  }
  const state = getState();
  if ((0, _selectors.isMetadataTransformStatsLoading)(state)) return;
  dispatch({
    type: 'metadataTransformStatsChanged',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.getMetadataTransformStats)(state)))
  });
  try {
    const transformStatsResponse = await http.get(_constants.METADATA_TRANSFORMS_STATUS_INTERNAL_ROUTE, {
      version: '1'
    });
    dispatch({
      type: 'metadataTransformStatsChanged',
      payload: (0, _state.createLoadedResourceState)(transformStatsResponse.transforms)
    });
  } catch (error) {
    dispatch({
      type: 'metadataTransformStatsChanged',
      payload: (0, _state.createFailedResourceState)(error)
    });
  }
}
async function fetchNonExistingPolicies({
  store,
  hosts,
  coreStart
}) {
  if (!(0, _authz.canFetchPackageAndAgentPolicies)(coreStart.application.capabilities)) {
    return;
  }
  const {
    getState,
    dispatch
  } = store;
  try {
    const missingPolicies = await getNonExistingPoliciesForEndpointList(coreStart.http, hosts, (0, _selectors.nonExistingPolicies)(getState()));
    if (missingPolicies !== undefined) {
      dispatch({
        type: 'serverReturnedEndpointNonExistingPolicies',
        payload: missingPolicies
      });
    }
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    // Ignore Errors, since this should not hinder the user's ability to use the UI
    logError(error);
  }
}