"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchEntityDetailsHighlights = void 0;
var _react = require("react");
var _inferenceCommon = require("@kbn/inference-common");
var _i18n = require("@kbn/i18n");
var _kibana_react = require("../../../../common/lib/kibana/kibana_react");
var _use_global_time = require("../../../../common/containers/use_global_time");
var _use_app_toasts = require("../../../../common/hooks/use_app_toasts");
var _api = require("../../../api/api");
var _helpers = require("../utils/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const entityHighlightsSchema = {
  type: 'object',
  properties: {
    highlights: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          title: {
            type: 'string',
            description: 'The title of the highlight section'
          },
          text: {
            type: 'string',
            description: 'The detailed text content for this highlight section.'
          }
        },
        required: ['title', 'text']
      },
      description: 'A list of highlight items, each with a title and text. Only include highlights for which information is available in the context.'
    },
    recommendedActions: {
      type: 'array',
      items: {
        type: 'string'
      },
      description: 'A list of actionable recommendations for the security analyst. Omit this field if no actions are available.'
    }
  },
  required: ['highlights']
};
const useFetchEntityDetailsHighlights = ({
  connectorId,
  anonymizationFields,
  entityType,
  entityIdentifier
}) => {
  const {
    inference
  } = (0, _kibana_react.useKibana)().services;
  const {
    fetchEntityDetailsHighlights
  } = (0, _api.useEntityAnalyticsRoutes)();
  const {
    addError
  } = (0, _use_app_toasts.useAppToasts)();
  const {
    from,
    to
  } = (0, _use_global_time.useGlobalTime)();
  const [isChatLoading, setIsChatLoading] = (0, _react.useState)(false);
  const [abortController, setAbortController] = (0, _react.useState)(null);
  const [error, setError] = (0, _react.useState)(null);
  const [assistantResult, setAssistantResult] = (0, _react.useState)(null);
  const fetchEntityHighlights = (0, _react.useCallback)(async () => {
    const errorTitle = _i18n.i18n.translate('xpack.securitySolution.flyout.entityDetails.highlights.fetch.errorTitle', {
      defaultMessage: `Failed to run LLM`
    });

    // Clear any previously shown error while a new generation attempt is in progress
    setError(null);
    const {
      summary,
      replacements,
      prompt
    } = await fetchEntityDetailsHighlights({
      entityType,
      entityIdentifier,
      anonymizationFields,
      from: new Date(from).getTime(),
      to: new Date(to).getTime(),
      connectorId
    }).catch(e => {
      const caughtError = e instanceof Error ? e : new Error(String(e));
      addError(caughtError, {
        title: errorTitle
      });
      setError(caughtError);
      return {
        summary: null,
        replacements: null,
        prompt: null
      };
    });
    if (!summary || !replacements || !prompt) {
      return;
    }
    const summaryFormatted = JSON.stringify(summary);
    const controller = new AbortController();
    setAbortController(controller);
    setIsChatLoading(true);
    try {
      const outputResponse = await inference.output({
        id: 'entity-highlights',
        connectorId,
        schema: entityHighlightsSchema,
        system: prompt,
        input: `Context:
            EntityType: ${entityType},
            EntityIdentifier: ${(0, _helpers.getAnonymizedEntityIdentifier)(entityIdentifier, replacements)},
          ${summaryFormatted}`,
        abortSignal: controller.signal
      });
      const typedOutput = outputResponse.output;
      setAssistantResult({
        summaryAsText: summaryFormatted,
        response: typedOutput,
        replacements,
        generatedAt: Date.now()
      });
    } catch (e) {
      if ((0, _inferenceCommon.isInferenceRequestAbortedError)(e)) {
        return;
      }
      const caughtError = e instanceof Error ? e : new Error(String(e));
      addError(caughtError, {
        title: errorTitle
      });
      setError(caughtError);
    } finally {
      setIsChatLoading(false);
      setAbortController(null);
    }
  }, [fetchEntityDetailsHighlights, entityType, entityIdentifier, anonymizationFields, from, to, connectorId, inference, addError]);
  const abortStream = (0, _react.useCallback)(() => {
    if (abortController) {
      abortController.abort();
      setAbortController(null);
      setIsChatLoading(false);
    }
  }, [abortController]);
  return {
    fetchEntityHighlights,
    isChatLoading,
    abortStream,
    result: assistantResult,
    error
  };
};
exports.useFetchEntityDetailsHighlights = useFetchEntityDetailsHighlights;