"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bootstrapPrebuiltRules = exports.addRuleExceptions = void 0;
exports.bulkExportRules = bulkExportRules;
exports.patchRule = exports.importRules = exports.getPrebuiltRulesStatus = exports.getPrebuiltRuleBaseVersion = exports.getPrePackagedRulesStatus = exports.findRuleExceptionReferences = exports.fetchRulesSnoozeSettings = exports.fetchRules = exports.fetchRuleManagementFilters = exports.fetchRuleById = exports.fetchCoverageOverview = exports.fetchConnectors = exports.fetchConnectorTypes = exports.exportRules = exports.createRule = void 0;
exports.performBulkAction = performBulkAction;
exports.updateRule = exports.reviewRuleUpgrade = exports.reviewRuleInstall = exports.revertPrebuiltRule = exports.previewRule = exports.performUpgradeRules = exports.performInstallSpecificRules = exports.performInstallAllRules = void 0;
var _common = require("@kbn/alerting-plugin/common");
var _common2 = require("@kbn/actions-plugin/common");
var _rule_filtering = require("../../../../common/detection_engine/rule_management/rule_filtering");
var _prebuilt_rules = require("../../../../common/api/detection_engine/prebuilt_rules");
var _rule_management = require("../../../../common/api/detection_engine/rule_management");
var _constants = require("../../../../common/constants");
var _rule_exceptions = require("../../../../common/api/detection_engine/rule_exceptions");
var _kibana = require("../../../common/lib/kibana");
var i18n = _interopRequireWildcard(require("../../common/translations"));
var _utils = require("../../rule_gaps/api/hooks/utils");
var _constants2 = require("../../rule_gaps/constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Create provided Rule
 *
 * @param rule RuleCreateProps to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const createRule = async ({
  rule,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_URL, {
  method: 'POST',
  version: '2023-10-31',
  body: JSON.stringify(rule),
  signal
});

/**
 * Update provided Rule using PUT
 *
 * @param rule RuleUpdateProps to be updated
 * @param signal to cancel request
 *
 * @returns Promise<RuleResponse> An updated rule
 *
 * In fact this function should return Promise<RuleResponse> but it'd require massive refactoring.
 * It should be addressed as a part of OpenAPI schema adoption.
 *
 * @throws An error if response is not OK
 */
exports.createRule = createRule;
const updateRule = async ({
  rule,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_URL, {
  method: 'PUT',
  version: '2023-10-31',
  body: JSON.stringify(rule),
  signal
});

/**
 * Patch provided rule
 * NOTE: The rule edit flow does NOT use patch as it relies on the
 * functionality of PUT to delete field values when not provided, if
 * just expecting changes, use this `patchRule`
 *
 * @param ruleProperties to patch
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.updateRule = updateRule;
const patchRule = async ({
  ruleProperties,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_URL, {
  method: 'PATCH',
  version: '2023-10-31',
  body: JSON.stringify(ruleProperties),
  signal
});

/**
 * Preview provided Rule
 *
 * @param rule RuleCreateProps to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.patchRule = patchRule;
const previewRule = async ({
  rule,
  enableLoggedRequests,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_PREVIEW, {
  method: 'POST',
  version: '2023-10-31',
  body: JSON.stringify(rule),
  signal,
  query: enableLoggedRequests ? {
    enable_logged_requests: enableLoggedRequests
  } : undefined
});

/**
 * Fetches all rules from the Detection Engine API
 *
 * @param filterOptions desired filters (e.g. filter/sortField/sortOrder)
 * @param pagination desired pagination options (e.g. page/perPage)
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.previewRule = previewRule;
const fetchRules = async ({
  filterOptions = {
    filter: '',
    showCustomRules: false,
    showElasticRules: false,
    tags: []
  },
  sortingOptions = {
    field: 'enabled',
    order: 'desc'
  },
  pagination = {
    page: 1,
    perPage: 20
  },
  signal
}) => {
  var _filterOptions$gapFil, _filterOptions$gapFil2;
  const kql = (0, _rule_filtering.convertRulesFilterToKQL)(filterOptions);
  const shouldApplyDefaultGapsRange = Boolean(filterOptions === null || filterOptions === void 0 ? void 0 : (_filterOptions$gapFil = filterOptions.gapFillStatuses) === null || _filterOptions$gapFil === void 0 ? void 0 : _filterOptions$gapFil.length);
  const defaultGapsRange = shouldApplyDefaultGapsRange ? (0, _utils.getGapRange)(_constants2.defaultRangeValue) : undefined;
  const query = {
    page: pagination.page,
    per_page: pagination.perPage,
    sort_field: sortingOptions.field,
    sort_order: sortingOptions.order,
    ...(filterOptions !== null && filterOptions !== void 0 && (_filterOptions$gapFil2 = filterOptions.gapFillStatuses) !== null && _filterOptions$gapFil2 !== void 0 && _filterOptions$gapFil2.length ? {
      gap_fill_statuses: filterOptions.gapFillStatuses
    } : {}),
    ...(defaultGapsRange ? {
      gaps_range_start: defaultGapsRange.start,
      gaps_range_end: defaultGapsRange.end
    } : {}),
    ...(kql !== '' ? {
      filter: kql
    } : {})
  };
  return _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_URL_FIND, {
    method: 'GET',
    version: '2023-10-31',
    query,
    signal
  });
};

/**
 * Fetch a Rule by providing a Rule ID
 *
 * @param id Rule ID's (not rule_id)
 * @param signal to cancel request
 *
 * @returns Promise<RuleResponse>
 *
 * In fact this function should return Promise<RuleResponse> but it'd require massive refactoring.
 * It should be addressed as a part of OpenAPI schema adoption.
 *
 * @throws An error if response is not OK
 */
exports.fetchRules = fetchRules;
const fetchRuleById = async ({
  id,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_URL, {
  method: 'GET',
  version: '2023-10-31',
  query: {
    id
  },
  signal
});

/**
 * Fetch rule snooze settings for each provided ruleId
 *
 * @param ids Rule IDs (not rule_id)
 * @param signal to cancel request
 *
 * @returns An error if response is not OK
 */
exports.fetchRuleById = fetchRuleById;
const fetchRulesSnoozeSettings = async ({
  ids,
  signal
}) => {
  var _response$data;
  const response = await _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_API_FIND_RULES_PATH, {
    method: 'POST',
    body: JSON.stringify({
      filter: ids.map(x => `alert.id:"alert:${x}"`).join(' or '),
      fields: JSON.stringify(['name', 'muteAll', 'activeSnoozes', 'isSnoozedUntil', 'snoozeSchedule']),
      per_page: ids.length
    }),
    signal
  });
  return (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.reduce((result, {
    id,
    ...snoozeSettings
  }) => {
    var _snoozeSettings$name, _snoozeSettings$mute_, _snoozeSettings$activ;
    result[id] = {
      name: (_snoozeSettings$name = snoozeSettings.name) !== null && _snoozeSettings$name !== void 0 ? _snoozeSettings$name : '',
      muteAll: (_snoozeSettings$mute_ = snoozeSettings.mute_all) !== null && _snoozeSettings$mute_ !== void 0 ? _snoozeSettings$mute_ : false,
      activeSnoozes: (_snoozeSettings$activ = snoozeSettings.active_snoozes) !== null && _snoozeSettings$activ !== void 0 ? _snoozeSettings$activ : [],
      isSnoozedUntil: snoozeSettings.is_snoozed_until ? new Date(snoozeSettings.is_snoozed_until) : undefined,
      snoozeSchedule: snoozeSettings.snooze_schedule
    };
    return result;
  }, {});
};
exports.fetchRulesSnoozeSettings = fetchRulesSnoozeSettings;
const fetchConnectors = signal => _kibana.KibanaServices.get().http.fetch(`${_common2.BASE_ACTION_API_PATH}/connectors`, {
  method: 'GET',
  signal
});
exports.fetchConnectors = fetchConnectors;
const fetchCoverageOverview = async ({
  filter,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_rule_management.RULE_MANAGEMENT_COVERAGE_OVERVIEW_URL, {
  method: 'POST',
  version: '1',
  body: JSON.stringify({
    filter
  }),
  signal
});
exports.fetchCoverageOverview = fetchCoverageOverview;
const fetchConnectorTypes = signal => _kibana.KibanaServices.get().http.fetch(`${_common2.BASE_ACTION_API_PATH}/connector_types`, {
  method: 'GET',
  signal,
  query: {
    feature_id: 'siem'
  }
});
exports.fetchConnectorTypes = fetchConnectorTypes;
/**
 * Perform bulk action with rules selected by a filter query
 *
 * @param bulkAction bulk action which contains type, query or ids and edit fields
 * @param dryRun enables dry run mode for bulk actions
 *
 * @throws An error if response is not OK
 */
async function performBulkAction({
  bulkAction,
  dryRun = false
}) {
  var _bulkAction$gapRange, _bulkAction$gapRange2, _bulkAction$gapFillSt;
  const params = {
    action: bulkAction.type,
    query: bulkAction.query,
    ids: bulkAction.ids,
    edit: bulkAction.type === _rule_management.BulkActionTypeEnum.edit ? bulkAction.editPayload : undefined,
    duplicate: bulkAction.type === _rule_management.BulkActionTypeEnum.duplicate ? bulkAction.duplicatePayload : undefined,
    run: bulkAction.type === _rule_management.BulkActionTypeEnum.run ? bulkAction.runPayload : undefined,
    gaps_range_start: 'gapRange' in bulkAction ? (_bulkAction$gapRange = bulkAction.gapRange) === null || _bulkAction$gapRange === void 0 ? void 0 : _bulkAction$gapRange.start : undefined,
    gaps_range_end: 'gapRange' in bulkAction ? (_bulkAction$gapRange2 = bulkAction.gapRange) === null || _bulkAction$gapRange2 === void 0 ? void 0 : _bulkAction$gapRange2.end : undefined,
    gap_fill_statuses: 'gapFillStatuses' in bulkAction && (_bulkAction$gapFillSt = bulkAction.gapFillStatuses) !== null && _bulkAction$gapFillSt !== void 0 && _bulkAction$gapFillSt.length ? bulkAction.gapFillStatuses : undefined,
    fill_gaps: bulkAction.type === _rule_management.BulkActionTypeEnum.fill_gaps ? bulkAction.fillGapsPayload : undefined
  };
  return _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_BULK_ACTION, {
    method: 'POST',
    version: '2023-10-31',
    body: JSON.stringify(params),
    query: {
      dry_run: dryRun
    }
  });
}
/**
 * Bulk export rules selected by a filter query
 *
 * @param queryOrIds filter query to select rules to perform bulk action with or rule ids to select rules to perform bulk action with
 *
 * @throws An error if response is not OK
 */
async function bulkExportRules(queryOrIds) {
  const params = {
    action: _rule_management.BulkActionTypeEnum.export,
    query: queryOrIds.query,
    ids: queryOrIds.ids
  };
  return _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_BULK_ACTION, {
    method: 'POST',
    version: '2023-10-31',
    body: JSON.stringify(params)
  });
}
/**
 * Imports rules in the same format as exported via the _export API
 *
 * @param fileToImport File to upload containing rules to import
 * @param overwrite whether or not to overwrite rules with the same ruleId
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
const importRules = async ({
  fileToImport,
  overwrite = false,
  overwriteExceptions = false,
  overwriteActionConnectors = false,
  signal
}) => {
  const formData = new FormData();
  formData.append('file', fileToImport);
  return _kibana.KibanaServices.get().http.fetch(_constants.DETECTION_ENGINE_RULES_IMPORT_URL, {
    method: 'POST',
    version: '2023-10-31',
    headers: {
      'Content-Type': undefined
    },
    query: {
      overwrite,
      overwrite_exceptions: overwriteExceptions,
      overwrite_action_connectors: overwriteActionConnectors
    },
    body: formData,
    signal
  });
};

/**
 * Export rules from the server as a file download
 *
 * @param excludeExportDetails whether or not to exclude additional details at bottom of exported file (defaults to false)
 * @param filename of exported rules. Be sure to include `.ndjson` extension! (defaults to localized `rules_export.ndjson`)
 * @param ruleIds array of rule_id's (not id!) to export (empty array exports _all_ rules)
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.importRules = importRules;
const exportRules = async ({
  excludeExportDetails = false,
  filename = `${i18n.EXPORT_FILENAME}.ndjson`,
  ids = [],
  signal
}) => {
  const body = ids.length > 0 ? JSON.stringify({
    objects: ids.map(rule => ({
      rule_id: rule
    }))
  }) : undefined;
  return _kibana.KibanaServices.get().http.fetch(`${_constants.DETECTION_ENGINE_RULES_URL}/_export`, {
    method: 'POST',
    version: '2023-10-31',
    body,
    query: {
      exclude_export_details: excludeExportDetails,
      file_name: filename
    },
    signal
  });
};

/**
 * Fetch rule filters related information like installed rules count, tags and etc
 *
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.exportRules = exportRules;
const fetchRuleManagementFilters = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_rule_management.RULE_MANAGEMENT_FILTERS_URL, {
  method: 'GET',
  version: '1',
  signal
});

/**
 * Get pre packaged rules Status
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.fetchRuleManagementFilters = fetchRuleManagementFilters;
const getPrePackagedRulesStatus = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.PREBUILT_RULES_STATUS_URL, {
  method: 'GET',
  version: '2023-10-31',
  signal
});

/**
 * Fetch info on what exceptions lists are referenced by what rules
 *
 * @param lists exception list information needed for making request
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.getPrePackagedRulesStatus = getPrePackagedRulesStatus;
const findRuleExceptionReferences = async ({
  lists,
  signal
}) => {
  const idsUndefined = lists.some(({
    id
  }) => id === undefined);
  const query = idsUndefined ? {
    namespace_types: lists.map(({
      namespaceType
    }) => namespaceType).join(',')
  } : {
    ids: lists.map(({
      id
    }) => id).join(','),
    list_ids: lists.map(({
      listId
    }) => listId).join(','),
    namespace_types: lists.map(({
      namespaceType
    }) => namespaceType).join(',')
  };
  return _kibana.KibanaServices.get().http.fetch(_rule_exceptions.DETECTION_ENGINE_RULES_EXCEPTIONS_REFERENCE_URL, {
    method: 'GET',
    version: '1',
    query,
    signal
  });
};

/**
 * Add exception items to default rule exception list
 *
 * @param ruleId `id` of rule to add items to
 * @param items CreateRuleExceptionListItemSchema[]
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.findRuleExceptionReferences = findRuleExceptionReferences;
const addRuleExceptions = async ({
  ruleId,
  items,
  signal
}) => _kibana.KibanaServices.get().http.fetch(`${_constants.DETECTION_ENGINE_RULES_URL}/${ruleId}/exceptions`, {
  method: 'POST',
  version: '2023-10-31',
  body: JSON.stringify({
    items
  }),
  signal
});

/**
 * NEW PREBUILT RULES ROUTES START HERE! 👋
 * USE THESE ONES! THEY'RE THE NICE ONES, PROMISE!
 */

/**
 * Get prebuilt rules status
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.addRuleExceptions = addRuleExceptions;
const getPrebuiltRulesStatus = async ({
  signal
}) => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.GET_PREBUILT_RULES_STATUS_URL, {
  method: 'GET',
  version: '1',
  signal
});

/**
 * Review prebuilt rules upgrade
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.getPrebuiltRulesStatus = getPrebuiltRulesStatus;
const reviewRuleUpgrade = async ({
  signal,
  request
}) => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.REVIEW_RULE_UPGRADE_URL, {
  method: 'POST',
  version: '1',
  signal,
  body: JSON.stringify(request)
});

/**
 * Review prebuilt rules install (new rules)
 *
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.reviewRuleUpgrade = reviewRuleUpgrade;
const reviewRuleInstall = async ({
  signal,
  request
}) => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.REVIEW_RULE_INSTALLATION_URL, {
  method: 'POST',
  version: '1',
  signal,
  body: JSON.stringify(request)
});
exports.reviewRuleInstall = reviewRuleInstall;
const performInstallAllRules = async () => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.PERFORM_RULE_INSTALLATION_URL, {
  method: 'POST',
  version: '1',
  body: JSON.stringify({
    mode: 'ALL_RULES'
  })
});
exports.performInstallAllRules = performInstallAllRules;
const performInstallSpecificRules = async rules => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.PERFORM_RULE_INSTALLATION_URL, {
  method: 'POST',
  version: '1',
  body: JSON.stringify({
    mode: 'SPECIFIC_RULES',
    rules
  })
});
exports.performInstallSpecificRules = performInstallSpecificRules;
const performUpgradeRules = async body => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.PERFORM_RULE_UPGRADE_URL, {
  method: 'POST',
  version: '1',
  body: JSON.stringify(body)
});
exports.performUpgradeRules = performUpgradeRules;
const bootstrapPrebuiltRules = async () => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.BOOTSTRAP_PREBUILT_RULES_URL, {
  method: 'POST',
  version: '1'
});
exports.bootstrapPrebuiltRules = bootstrapPrebuiltRules;
const getPrebuiltRuleBaseVersion = async ({
  signal,
  request
}) => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.GET_PREBUILT_RULES_BASE_VERSION_URL, {
  method: 'GET',
  version: '1',
  signal,
  query: request
});
exports.getPrebuiltRuleBaseVersion = getPrebuiltRuleBaseVersion;
const revertPrebuiltRule = async body => _kibana.KibanaServices.get().http.fetch(_prebuilt_rules.REVERT_PREBUILT_RULES_URL, {
  method: 'POST',
  version: '1',
  body: JSON.stringify(body)
});
exports.revertPrebuiltRule = revertPrebuiltRule;