"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGapAutoFillScheduler = exports.scheduleRuleRun = exports.getRuleIdsWithGaps = exports.getGapsSummaryByRuleIds = exports.getGapAutoFillScheduler = exports.findGapsForRule = exports.findGapAutoFillSchedulerLogs = exports.findBackfillsForRules = exports.fillGapByIdForRule = exports.deleteBackfill = exports.createGapAutoFillScheduler = void 0;
var _common = require("@kbn/alerting-plugin/common");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _kibana = require("../../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Schedule rules run over a specified time range
 *
 * @param ruleIds `rule_id`s of each rule to be backfilled
 * @param timeRange the time range over which the backfill should apply
 *
 * @throws An error if response is not OK
 */
const scheduleRuleRun = async ({
  ruleIds,
  timeRange
}) => {
  const params = ruleIds.map(ruleId => {
    return {
      rule_id: ruleId,
      ranges: [{
        start: timeRange.startDate.toISOString(),
        end: timeRange.endDate.toISOString()
      }]
    };
  });
  return _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_BACKFILL_SCHEDULE_API_PATH, {
    method: 'POST',
    version: '2023-10-31',
    body: JSON.stringify(params)
  });
};

/**
 * Find backfills for the given rule IDs
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.scheduleRuleRun = scheduleRuleRun;
const findBackfillsForRules = async ({
  ruleIds,
  page,
  perPage,
  signal,
  sortField = 'createdAt',
  sortOrder = 'desc',
  initiator
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_BACKFILL_FIND_API_PATH, {
  method: 'POST',
  query: {
    ...(ruleIds && ruleIds.length > 0 ? {
      rule_ids: ruleIds.join(',')
    } : {}),
    page,
    per_page: perPage,
    sort_field: sortField,
    sort_order: sortOrder,
    initiator
  },
  signal
});

/**
 * Delete backfill by ID
 * @param backfillId
 * @returns
 */
exports.findBackfillsForRules = findBackfillsForRules;
const deleteBackfill = async ({
  backfillId
}) => {
  return _kibana.KibanaServices.get().http.fetch(`${_common.INTERNAL_ALERTING_BACKFILL_API_PATH}/${backfillId}`, {
    method: 'DELETE'
  });
};

/**
 * Find rules with gaps
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.deleteBackfill = deleteBackfill;
const getRuleIdsWithGaps = async ({
  signal,
  start,
  end,
  gapFillStatuses = [_common.gapFillStatus.UNFILLED],
  hasUnfilledIntervals,
  hasInProgressIntervals,
  hasFilledIntervals
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_GET_RULES_API_PATH, {
  method: 'POST',
  body: JSON.stringify({
    start,
    end,
    highest_priority_gap_fill_statuses: gapFillStatuses,
    ...(hasUnfilledIntervals !== undefined && {
      has_unfilled_intervals: hasUnfilledIntervals
    }),
    ...(hasInProgressIntervals !== undefined && {
      has_in_progress_intervals: hasInProgressIntervals
    }),
    ...(hasFilledIntervals !== undefined && {
      has_filled_intervals: hasFilledIntervals
    })
  }),
  signal
});

/**
 * Find total gap info for the given rule IDs
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.getRuleIdsWithGaps = getRuleIdsWithGaps;
const getGapsSummaryByRuleIds = async ({
  signal,
  start,
  end,
  ruleIds
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_GET_SUMMARY_BY_RULE_IDS_API_PATH, {
  signal,
  method: 'POST',
  body: JSON.stringify({
    start,
    end,
    rule_ids: ruleIds
  })
});

/**
 * Find gaps for the given rule ID
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.getGapsSummaryByRuleIds = getGapsSummaryByRuleIds;
const findGapsForRule = async ({
  ruleId,
  page,
  perPage,
  signal,
  sortField = '@timestamp',
  sortOrder = 'desc',
  start,
  end,
  statuses
}) => {
  const startDate = _datemath.default.parse(start);
  const endDate = _datemath.default.parse(end, {
    roundUp: true
  });
  return _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_FIND_API_PATH, {
    method: 'POST',
    body: JSON.stringify({
      rule_id: ruleId,
      page,
      per_page: perPage,
      sort_field: sortField,
      sort_order: sortOrder,
      start: startDate === null || startDate === void 0 ? void 0 : startDate.utc().toISOString(),
      end: endDate === null || endDate === void 0 ? void 0 : endDate.utc().toISOString(),
      statuses
    }),
    signal
  });
};

/**
 * Fill gap by Id for the given rule ID
 * @param ruleIds string[]
 * @param signal? AbortSignal
 * @returns
 */
exports.findGapsForRule = findGapsForRule;
const fillGapByIdForRule = async ({
  ruleId,
  gapId,
  signal
}) => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_FILL_BY_ID_API_PATH, {
  method: 'POST',
  query: {
    rule_id: ruleId,
    gap_id: gapId
  },
  signal
});
exports.fillGapByIdForRule = fillGapByIdForRule;
const getGapAutoFillScheduler = async ({
  id,
  signal
}) => _kibana.KibanaServices.get().http.fetch(`${_common.INTERNAL_ALERTING_GAPS_AUTO_FILL_SCHEDULER_API_PATH}/${id}`, {
  method: 'GET',
  signal
});
exports.getGapAutoFillScheduler = getGapAutoFillScheduler;
const createGapAutoFillScheduler = async params => _kibana.KibanaServices.get().http.fetch(_common.INTERNAL_ALERTING_GAPS_AUTO_FILL_SCHEDULER_API_PATH, {
  method: 'POST',
  body: JSON.stringify({
    id: params.id,
    name: params.name,
    scope: params.scope,
    schedule: params.schedule,
    rule_types: params.ruleTypes,
    gap_fill_range: params.gapFillRange,
    max_backfills: params.maxBackfills,
    num_retries: params.numRetries,
    enabled: params.enabled
  })
});
exports.createGapAutoFillScheduler = createGapAutoFillScheduler;
const updateGapAutoFillScheduler = async params => _kibana.KibanaServices.get().http.fetch(`${_common.INTERNAL_ALERTING_GAPS_AUTO_FILL_SCHEDULER_API_PATH}/${params.id}`, {
  method: 'PUT',
  body: JSON.stringify({
    name: params.name,
    scope: params.scope,
    schedule: params.schedule,
    rule_types: params.ruleTypes,
    gap_fill_range: params.gapFillRange,
    max_backfills: params.maxBackfills,
    num_retries: params.numRetries,
    enabled: params.enabled
  })
});
exports.updateGapAutoFillScheduler = updateGapAutoFillScheduler;
const findGapAutoFillSchedulerLogs = async ({
  id,
  start,
  end,
  page,
  perPage,
  sortField,
  sortDirection,
  statuses,
  signal
}) => _kibana.KibanaServices.get().http.fetch(`${_common.INTERNAL_ALERTING_GAPS_AUTO_FILL_SCHEDULER_API_PATH}/${id}/logs`, {
  method: 'POST',
  signal,
  body: JSON.stringify({
    start,
    end,
    page,
    per_page: perPage,
    sort_field: sortField,
    sort_direction: sortDirection,
    ...(statuses && statuses.length > 0 && {
      statuses: [...statuses]
    })
  })
});
exports.findGapAutoFillSchedulerLogs = findGapAutoFillSchedulerLogs;