"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useInitDataViewManager = void 0;
var _reactRedux = require("react-redux");
var _react = require("react");
var _toolkit = require("@reduxjs/toolkit");
var _constants = require("../../../common/constants");
var _kibana = require("../../common/lib/kibana");
var _data_view_selected = require("../redux/listeners/data_view_selected");
var _init_listener = require("../redux/listeners/init_listener");
var _slices = require("../redux/slices");
var _constants2 = require("../constants");
var _use_experimental_features = require("../../common/hooks/use_experimental_features");
var _user_info = require("../../detections/components/user_info");
var _use_data_view_logger = require("./use_data_view_logger");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addListener = listener => (0, _toolkit.addListener)(listener);
const removeListener = listener => (0, _toolkit.removeListener)(listener);

/**
 * Should only be used once in the application, on the top level of the rendering tree
 */
const useInitDataViewManager = () => {
  const dispatch = (0, _reactRedux.useDispatch)();
  const services = (0, _kibana.useKibana)().services;
  const newDataViewPickerEnabled = (0, _use_experimental_features.useIsExperimentalFeatureEnabled)('newDataViewPickerEnabled');
  const attacksAlertsAlignmentEnabled = services.featureFlags.getBooleanValue(_constants.ATTACKS_ALERTS_ALIGNMENT_ENABLED, false);
  const logger = (0, _use_data_view_logger.useDataViewManagerLogger)('useInitDataViewManager');
  const createInitListenerLogger = (0, _use_data_view_logger.useDataViewManagerLogger)('createInitListener');
  const createDataViewSelectedListenerLogger = (0, _use_data_view_logger.useDataViewManagerLogger)('createDataViewSelectedListener');
  const {
    loading: loadingSignalIndex,
    signalIndexName,
    signalIndexMappingOutdated
  } = (0, _user_info.useUserInfo)();
  const onSignalIndexUpdated = (0, _react.useCallback)(() => {
    if (!loadingSignalIndex && signalIndexName != null) {
      dispatch(_slices.sharedDataViewManagerSlice.actions.setSignalIndex({
        name: signalIndexName,
        isOutdated: !!signalIndexMappingOutdated
      }));
      logger.debug(`signal index set: ${signalIndexName}`);
    }
  }, [dispatch, loadingSignalIndex, logger, signalIndexMappingOutdated, signalIndexName]);
  (0, _react.useEffect)(() => {
    // TODO: (new data view picker) remove this in cleanup phase https://github.com/elastic/security-team/issues/12665
    // Also, make sure it works exactly as x-pack/solutions/security/plugins/security_solution/public/sourcerer/containers/use_init_sourcerer.tsx
    if (!newDataViewPickerEnabled) {
      return;
    }
    onSignalIndexUpdated();
    // because we only want onSignalIndexUpdated to run when signalIndexName updates,
    // but we want to know about the updates from the dependencies of onSignalIndexUpdated
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [signalIndexName]);
  (0, _react.useEffect)(() => {
    // TODO: (new data view picker) remove this in cleanup phase https://github.com/elastic/security-team/issues/12665
    if (!newDataViewPickerEnabled) {
      return;
    }

    // NOTE: init listener contains logic that preloads default security solution data view
    const dataViewsLoadingListener = (0, _init_listener.createInitListener)({
      dataViews: services.dataViews,
      http: services.http,
      uiSettings: services.uiSettings,
      application: services.application,
      spaces: services.spaces,
      storage: services.storage,
      logger: createInitListenerLogger
    }, attacksAlertsAlignmentEnabled);
    logger.debug('Registering data view manager listeners');
    dispatch(addListener(dataViewsLoadingListener));

    // NOTE: Every scope has its own listener instance; this allows for cancellation
    const listeners = [_constants2.PageScope.default, _constants2.PageScope.timeline, _constants2.PageScope.alerts, _constants2.PageScope.attacks, _constants2.PageScope.analyzer, _constants2.PageScope.explore].map(scope => (0, _data_view_selected.createDataViewSelectedListener)({
      scope,
      dataViews: services.dataViews,
      storage: services.storage,
      logger: createDataViewSelectedListenerLogger
    }));
    logger.debug('Registering data view selected listeners for all scopes');
    listeners.forEach(dataViewSelectedListener => {
      dispatch(addListener(dataViewSelectedListener));
    });

    // NOTE: this kicks off the data loading in the Data View Picker

    return () => {
      logger.debug('Cleaning up listeners');
      dispatch(removeListener(dataViewsLoadingListener));
      listeners.forEach(dataViewSelectedListener => {
        logger.debug(`Removed listener for scope ${dataViewSelectedListener.actionCreator.name}`);
        dispatch(removeListener(dataViewSelectedListener));
      });
    };
  }, [attacksAlertsAlignmentEnabled, dispatch, logger, newDataViewPickerEnabled, services.application, services.dataViews, services.http, createInitListenerLogger, services.spaces, services.storage, services.uiSettings, createDataViewSelectedListenerLogger]);
  return (0, _react.useCallback)(initialSelection => {
    logger.debug(`Initializing Data View Manager with initial selection: ${JSON.stringify(initialSelection)}`);
    dispatch(_slices.sharedDataViewManagerSlice.actions.init(initialSelection));
  }, [dispatch, logger]);
};
exports.useInitDataViewManager = useInitDataViewManager;