"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchFilter = exports.getStablePatternTitles = exports.filterJobs = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a filtered array of Jobs according to JobsTableFilters selections
 *
 * @param jobs to filter
 * @param selectedGroups groups to filter on
 * @param showCustomJobs whether or not to show all Custom Jobs (Non-embedded Jobs in SIEM Group)
 * @param showElasticJobs whether or not to show Elastic Jobs (Embedded ConfigTemplate Jobs)
 * @param filterQuery user-provided search string to filter for occurrence in job names/description
 */
const filterJobs = ({
  jobs,
  selectedGroups,
  showCustomJobs,
  showElasticJobs,
  filterQuery
}) => searchFilter(jobs.filter(job => !showCustomJobs || showCustomJobs && !job.isElasticJob).filter(job => !showElasticJobs || showElasticJobs && job.isElasticJob).filter(job => selectedGroups.length === 0 || selectedGroups.some(g => job.groups.includes(g))), filterQuery);

/**
 * Returns filtered array of Jobs based on user-provided search string to filter for occurrence in job names/description
 *
 * @param jobs to filter
 * @param filterQuery user-provided search string to filter for occurrence in job names/description
 */
exports.filterJobs = filterJobs;
const searchFilter = (jobs, filterQuery) => {
  const lowerCaseFilterQuery = filterQuery === null || filterQuery === void 0 ? void 0 : filterQuery.toLowerCase();
  return jobs.filter(job => {
    var _job$customSettings, _job$customSettings$s, _job$customSettings$s2;
    return lowerCaseFilterQuery == null ? true : job.id.includes(lowerCaseFilterQuery) || ((_job$customSettings = job.customSettings) === null || _job$customSettings === void 0 ? void 0 : (_job$customSettings$s = _job$customSettings.security_app_display_name) === null || _job$customSettings$s === void 0 ? void 0 : (_job$customSettings$s2 = _job$customSettings$s.toLowerCase()) === null || _job$customSettings$s2 === void 0 ? void 0 : _job$customSettings$s2.includes(lowerCaseFilterQuery)) || job.description.toLowerCase().includes(lowerCaseFilterQuery);
  });
};

/**
 * Given an array of titles this will always return the same string for usage within
 * useEffect and other shallow compare areas.
 * This won't return a stable reference for case sensitive strings intentionally for speed.
 * @param patterns string[] string array that will return a stable reference regardless of ordering or case sensitivity.
 */
exports.searchFilter = searchFilter;
const getStablePatternTitles = patterns => patterns.sort().join();
exports.getStablePatternTitles = getStablePatternTitles;