"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.waitForMetadataTransformsReady = exports.stopMetadataTransforms = exports.startMetadataTransforms = void 0;
var _package_v = require("./package_v2");
var _usage_tracker = require("../data_loaders/usage_tracker");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const waitForMetadataTransformsReady = exports.waitForMetadataTransformsReady = _usage_tracker.usageTracker.track('waitForMetadataTransformsReady', async (esClient, version) => {
  await waitFor(() => areMetadataTransformsReady(esClient, version));
});
const stopMetadataTransforms = exports.stopMetadataTransforms = _usage_tracker.usageTracker.track('stopMetadataTransforms', async (esClient, version) => {
  const transformIds = await getMetadataTransformIds(esClient, version);
  await Promise.all(transformIds.map(transformId => esClient.transform.stopTransform({
    transform_id: transformId,
    force: true,
    wait_for_completion: true,
    allow_no_match: true
  })));
});
const startMetadataTransforms = exports.startMetadataTransforms = _usage_tracker.usageTracker.track('startMetadataTransforms', async (esClient, agentIds, version) => {
  const transformIds = await getMetadataTransformIds(esClient, version);
  const isV2 = (0, _package_v.isEndpointPackageV2)(version);
  const currentTransformPrefix = isV2 ? _constants.METADATA_CURRENT_TRANSFORM_V2 : _constants.metadataTransformPrefix;
  const currentTransformId = transformIds.find(transformId => transformId.startsWith(currentTransformPrefix));
  const unitedTransformPrefix = isV2 ? _constants.METADATA_UNITED_TRANSFORM_V2 : _constants.METADATA_UNITED_TRANSFORM;
  const unitedTransformId = transformIds.find(transformId => transformId.startsWith(unitedTransformPrefix));
  if (!currentTransformId || !unitedTransformId) {
    // eslint-disable-next-line no-console
    console.warn('metadata transforms not found, skipping transform start');
    return;
  }
  try {
    await esClient.transform.startTransform({
      transform_id: currentTransformId
    });
  } catch (err) {
    // ignore if transform already started
    if (err.statusCode !== 409) {
      throw err;
    }
  }
  if (agentIds.length > 0) {
    await waitForCurrentMetdataDocs(esClient, agentIds);
  }
  try {
    await esClient.transform.startTransform({
      transform_id: unitedTransformId
    });
  } catch (err) {
    // ignore if transform already started
    if (err.statusCode !== 409) {
      throw err;
    }
  }
});
async function getMetadataTransformStats(esClient, /** The version of the Endpoint Package */
version) {
  const transformId = (0, _package_v.isEndpointPackageV2)(version) ? _constants.METADATA_TRANSFORMS_PATTERN_V2 : _constants.METADATA_TRANSFORMS_PATTERN;
  return (await esClient.transform.getTransformStats({
    transform_id: transformId,
    allow_no_match: true
  })).transforms;
}
async function getMetadataTransformIds(esClient, /** The version of the Endpoint Package */
version) {
  return (await getMetadataTransformStats(esClient, version)).map(transform => transform.id);
}
async function areMetadataTransformsReady(esClient, version) {
  const transforms = await getMetadataTransformStats(esClient, version);
  return !transforms.some(
  // TODO TransformGetTransformStatsTransformStats type needs to be updated to include health
  transform => {
    var _transform$health;
    return (transform === null || transform === void 0 ? void 0 : (_transform$health = transform.health) === null || _transform$health === void 0 ? void 0 : _transform$health.status) !== 'green';
  });
}
async function waitForCurrentMetdataDocs(esClient, agentIds) {
  var _agentIds$length;
  const query = agentIds.length ? {
    bool: {
      filter: [{
        terms: {
          'agent.id': agentIds
        }
      }]
    }
  } : {
    match_all: {}
  };
  const size = (_agentIds$length = agentIds.length) !== null && _agentIds$length !== void 0 ? _agentIds$length : 1;
  await waitFor(async () => (await esClient.search({
    index: _constants.metadataCurrentIndexPattern,
    query,
    size,
    rest_total_hits_as_int: true
  })).hits.total === size);
}
async function waitFor(cb, interval = 20000, maxAttempts = 6) {
  let attempts = 0;
  let isReady = false;
  while (!isReady) {
    await new Promise(res => setTimeout(() => res(''), interval));
    isReady = await cb();
    attempts++;
    if (attempts > maxAttempts) {
      return;
    }
  }
}