"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setArtifactOwnerSpaceId = exports.isTrustedProcessDescendantsTag = exports.isProcessDescendantsEnabled = exports.isPolicySelectionTag = exports.isOwnerSpaceIdTag = exports.isFilterProcessDescendantsTag = exports.isArtifactGlobal = exports.isArtifactByPolicy = exports.isAdvancedModeTag = exports.isAdvancedModeEnabled = exports.hasGlobalOrPerPolicyTag = exports.hasArtifactOwnerSpaceId = exports.getPolicyIdsFromArtifact = exports.getEffectedPolicySelectionByTags = exports.getArtifactTagsByPolicySelection = exports.getArtifactOwnerSpaceIds = exports.createExceptionListItemForCreate = exports.buildSpaceOwnerIdTag = exports.buildPerPolicyTag = void 0;
var _uuid = require("uuid");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLICY_ID_START_POSITION = _constants.BY_POLICY_ARTIFACT_TAG_PREFIX.length;
const isArtifactGlobal = item => {
  var _item$tags;
  return ((_item$tags = item.tags) !== null && _item$tags !== void 0 ? _item$tags : []).includes(_constants.GLOBAL_ARTIFACT_TAG);
};
exports.isArtifactGlobal = isArtifactGlobal;
const isArtifactByPolicy = item => {
  return !isArtifactGlobal(item);
};
exports.isArtifactByPolicy = isArtifactByPolicy;
const hasGlobalOrPerPolicyTag = item => {
  var _item$tags2;
  return ((_item$tags2 = item.tags) !== null && _item$tags2 !== void 0 ? _item$tags2 : []).some(tag => tag === _constants.GLOBAL_ARTIFACT_TAG || tag.startsWith(_constants.BY_POLICY_ARTIFACT_TAG_PREFIX));
};
exports.hasGlobalOrPerPolicyTag = hasGlobalOrPerPolicyTag;
const getPolicyIdsFromArtifact = item => {
  var _item$tags3;
  const policyIds = [];
  const tags = (_item$tags3 = item.tags) !== null && _item$tags3 !== void 0 ? _item$tags3 : [];
  for (const tag of tags) {
    if (tag !== _constants.GLOBAL_ARTIFACT_TAG && tag.startsWith(_constants.BY_POLICY_ARTIFACT_TAG_PREFIX)) {
      policyIds.push(tag.substring(POLICY_ID_START_POSITION));
    }
  }
  return policyIds;
};

/**
 * Given an Artifact tag value, utility will return a boolean indicating if that tag is
 * tracking artifact assignment (global/per-policy)
 */
exports.getPolicyIdsFromArtifact = getPolicyIdsFromArtifact;
const isPolicySelectionTag = tag => tag.startsWith(_constants.BY_POLICY_ARTIFACT_TAG_PREFIX) || tag === _constants.GLOBAL_ARTIFACT_TAG;

/**
 * Builds the per-policy tag that should be stored in the artifact's `tags` array
 * @param policyId
 */
exports.isPolicySelectionTag = isPolicySelectionTag;
const buildPerPolicyTag = policyId => {
  return `${_constants.BY_POLICY_ARTIFACT_TAG_PREFIX}${policyId}`;
};

/**
 * Return a list of artifact policy tags based on a current
 * selection by the EffectedPolicySelection component.
 */
exports.buildPerPolicyTag = buildPerPolicyTag;
const getArtifactTagsByPolicySelection = selection => {
  if (selection.isGlobal) {
    return [_constants.GLOBAL_ARTIFACT_TAG];
  }
  return selection.selected.map(policy => {
    return buildPerPolicyTag(policy.id);
  });
};

/**
 * Given a list of an Exception item tags it will return
 * the parsed policies from it.
 *
 * Policy tags follow the pattern `policy:id`
 * non policy tags will be ignored.
 */
exports.getArtifactTagsByPolicySelection = getArtifactTagsByPolicySelection;
const getEffectedPolicySelectionByTags = (tags, policies) => {
  if (tags.find(tag => tag === _constants.GLOBAL_ARTIFACT_TAG)) {
    return {
      isGlobal: true,
      selected: []
    };
  }
  const selected = tags.reduce((acc, tag) => {
    if (tag.startsWith(_constants.BY_POLICY_ARTIFACT_TAG_PREFIX)) {
      const id = tag.split(':')[1];
      const foundPolicy = policies.find(policy => policy.id === id);

      // edge case: a left over tag with a non-existed policy will be removed by verifying the policy exists
      if (foundPolicy !== undefined) {
        acc.push(foundPolicy);
      }
    }
    return acc;
  }, []);
  return {
    isGlobal: false,
    selected
  };
};
exports.getEffectedPolicySelectionByTags = getEffectedPolicySelectionByTags;
const isAdvancedModeEnabled = item => {
  var _item$tags4;
  return ((_item$tags4 = item.tags) !== null && _item$tags4 !== void 0 ? _item$tags4 : []).includes(_constants.ADVANCED_MODE_TAG);
};
exports.isAdvancedModeEnabled = isAdvancedModeEnabled;
const isAdvancedModeTag = tag => tag === _constants.ADVANCED_MODE_TAG;
exports.isAdvancedModeTag = isAdvancedModeTag;
const isProcessDescendantsEnabled = (item, tag = _constants.FILTER_PROCESS_DESCENDANTS_TAG) => {
  var _item$tags5;
  return ((_item$tags5 = item.tags) !== null && _item$tags5 !== void 0 ? _item$tags5 : []).includes(tag);
};

/** Checks if the given tag is for filtering process descendants in event filters */
exports.isProcessDescendantsEnabled = isProcessDescendantsEnabled;
const isFilterProcessDescendantsTag = tag => tag === _constants.FILTER_PROCESS_DESCENDANTS_TAG;

/** Checks if the given tag is for filtering process descendants in trusted apps */
exports.isFilterProcessDescendantsTag = isFilterProcessDescendantsTag;
const isTrustedProcessDescendantsTag = tag => tag === _constants.TRUSTED_PROCESS_DESCENDANTS_TAG;
exports.isTrustedProcessDescendantsTag = isTrustedProcessDescendantsTag;
const createExceptionListItemForCreate = listId => {
  return {
    comments: [],
    description: '',
    entries: [],
    item_id: undefined,
    list_id: listId,
    meta: {
      temporaryUuid: (0, _uuid.v4)()
    },
    name: '',
    namespace_type: 'agnostic',
    tags: [_constants.GLOBAL_ARTIFACT_TAG],
    type: 'simple',
    os_types: ['windows']
  };
};

/**
 * Checks the provided `tag` string to see if it is an owner apace ID tag
 * @param tag
 */
exports.createExceptionListItemForCreate = createExceptionListItemForCreate;
const isOwnerSpaceIdTag = tag => {
  return tag.startsWith(_constants.OWNER_SPACE_ID_TAG_PREFIX);
};

/**
 * Returns an array with all owner space IDs for the artifact
 */
exports.isOwnerSpaceIdTag = isOwnerSpaceIdTag;
const getArtifactOwnerSpaceIds = item => {
  var _item$tags6;
  return ((_item$tags6 = item.tags) !== null && _item$tags6 !== void 0 ? _item$tags6 : []).reduce((acc, tag) => {
    if (isOwnerSpaceIdTag(tag)) {
      acc.push(tag.substring(_constants.OWNER_SPACE_ID_TAG_PREFIX.length));
    }
    return acc;
  }, []);
};

/** Returns an Artifact `tag` value for a given space id */
exports.getArtifactOwnerSpaceIds = getArtifactOwnerSpaceIds;
const buildSpaceOwnerIdTag = spaceId => {
  if (spaceId.trim() === '') {
    throw new Error('spaceId must be a string with a length greater than zero.');
  }
  return `${_constants.OWNER_SPACE_ID_TAG_PREFIX}${spaceId}`;
};

/**
 * Sets the owner space id on the given artifact, if not already present.
 *
 * NOTE: this utility will mutate the artifact exception list item provided on input.
 *
 * @param item
 * @param spaceId
 */
exports.buildSpaceOwnerIdTag = buildSpaceOwnerIdTag;
const setArtifactOwnerSpaceId = (item, spaceId) => {
  if (spaceId.trim() === '') {
    throw new Error('spaceId must be a string with a length greater than zero.');
  }
  if (!getArtifactOwnerSpaceIds(item).includes(spaceId)) {
    if (!item.tags) {
      item.tags = [];
    }
    item.tags.push(buildSpaceOwnerIdTag(spaceId));
  }
};

/**
 * Checks to see if the artifact item has at least 1 owner space id tag
 * @param item
 */
exports.setArtifactOwnerSpaceId = setArtifactOwnerSpaceId;
const hasArtifactOwnerSpaceId = item => {
  var _item$tags7;
  return ((_item$tags7 = item.tags) !== null && _item$tags7 !== void 0 ? _item$tags7 : []).some(tag => isOwnerSpaceIdTag(tag));
};
exports.hasArtifactOwnerSpaceId = hasArtifactOwnerSpaceId;