"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexWorkflowInsights = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _workflow_insights = require("../types/workflow_insights");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexWorkflowInsights = async ({
  esClient,
  log,
  endpointId,
  os,
  count,
  antivirus,
  path
}) => {
  log.debug(`Indexing ${count} workflow insights`);
  const operations = Array.from({
    length: count
  }).flatMap((_, i) => {
    return [{
      index: {
        _index: '.edr-workflow-insights-default',
        op_type: 'create'
      }
    }, generateWorkflowInsightsDoc({
      endpointId,
      os,
      runNumber: i,
      antivirus,
      path
    })];
  });
  const response = await esClient.bulk({
    refresh: 'wait_for',
    operations
  });
  if (response.errors) {
    log.error(`There was an error indexing workflow insights ${JSON.stringify(response.items, null, 2)}`);
  } else {
    log.debug(`Indexed ${count} workflow insights successfully`);
  }
  return {
    data: response,
    cleanup: deleteIndexedWorkflowInsights.bind(null, esClient, response, log)
  };
};
exports.indexWorkflowInsights = indexWorkflowInsights;
const deleteIndexedWorkflowInsights = async (esClient, indexedWorkflowInsights, log) => {
  log.debug(`Deleting ${indexedWorkflowInsights.items.length} indexed workflow insights`);
  let response = {
    took: 0,
    errors: false,
    items: []
  };
  if (indexedWorkflowInsights.items.length) {
    const idsToDelete = indexedWorkflowInsights.items.filter(item => item.create).map(item => {
      var _item$create, _item$create2;
      return {
        delete: {
          _index: (_item$create = item.create) === null || _item$create === void 0 ? void 0 : _item$create._index,
          _id: (_item$create2 = item.create) === null || _item$create2 === void 0 ? void 0 : _item$create2._id
        }
      };
    });
    if (idsToDelete.length) {
      response = await esClient.bulk({
        operations: idsToDelete
      });
      log.debug('Indexed workflow insights deleted successfully');
    }
  }
  return {
    data: response
  };
};
const generateWorkflowInsightsDoc = ({
  endpointId,
  os,
  runNumber,
  antivirus,
  path
}) => {
  const currentTime = (0, _moment.default)();
  const signatureField = os === 'linux' ? undefined : os === 'windows' ? 'process.Ext.code_signature' : 'process.code_signature';
  const signatureValue = os === 'linux' ? undefined : 'Elastic';
  return {
    remediation: {
      exception_list_items: [{
        entries: [{
          field: 'process.executable.caseless',
          type: 'match',
          value: os !== 'windows' ? `/${runNumber}${path}` : (() => {
            const parts = path.split('\\'); // Split by Windows path separator
            const lastPart = parts.pop(); // Get the last part (executable)
            return `${parts.join('\\')}\\${runNumber}\\${lastPart}`; // Reconstruct the path
          })(),
          operator: 'included'
        }, ...(signatureField && signatureValue ? [{
          field: signatureField,
          operator: 'included',
          type: 'match',
          value: signatureValue
        }] : [])],
        list_id: 'endpoint_trusted_apps',
        name: `${antivirus}`,
        os_types: [os],
        description: 'Suggested by Automatic Troubleshooting',
        tags: ['policy:all']
      }]
    },
    metadata: {
      notes: {
        llm_model: ''
      },
      display_name: `${antivirus}`
    },
    '@timestamp': currentTime,
    action: {
      type: _workflow_insights.ActionType.Refreshed,
      timestamp: currentTime
    },
    source: {
      data_range_end: currentTime.clone().add(24, 'hours'),
      id: '7184ab52-c318-4c91-b765-805f889e34e2',
      type: _workflow_insights.SourceType.LlmConnector,
      data_range_start: currentTime
    },
    message: 'Incompatible antiviruses detected',
    category: _workflow_insights.Category.Endpoint,
    type: 'incompatible_antivirus',
    value: `${antivirus} ${path}${signatureValue ? ` ${signatureValue}` : ''}`,
    target: {
      ids: [endpointId],
      type: _workflow_insights.TargetType.Endpoint
    }
  };
};