"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexEndpointHostForPolicy = exports.indexEndpointHostDocs = exports.deleteIndexedEndpointHosts = exports.buildIndexHostsResponse = void 0;
var _lodash = require("lodash");
var _uuid = require("uuid");
var _common = require("@kbn/fleet-plugin/common");
var _transforms = require("../utils/transforms");
var _format_axios_error = require("../format_axios_error");
var _errors = require("../errors");
var _usage_tracker = require("./usage_tracker");
var _generate_data = require("../generate_data");
var _types = require("../types");
var _index_fleet_agent = require("./index_fleet_agent");
var _index_endpoint_fleet_actions = require("./index_endpoint_fleet_actions");
var _index_fleet_endpoint_policy = require("./index_fleet_endpoint_policy");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildIndexHostsResponse = () => {
  return {
    hosts: [],
    agents: [],
    policyResponses: [],
    metadataIndex: _constants.METADATA_DATASTREAM,
    policyResponseIndex: _constants.POLICY_RESPONSE_INDEX,
    fleetAgentsIndex: '',
    endpointActionResponses: [],
    endpointActionResponsesIndex: '',
    endpointActions: [],
    endpointActionsIndex: '',
    actionResponses: [],
    responsesIndex: '',
    actions: [],
    actionsIndex: '',
    integrationPolicies: [],
    agentPolicies: []
  };
};

/**
 * Indexes the requested number of documents for the endpoint host metadata currently being output by the generator.
 * Endpoint Host metadata documents are added to an index that is set as "append only", thus one Endpoint host could
 * have multiple documents in that index.
 *
 * @param numDocs
 * @param client
 * @param kbnClient
 * @param realPolicies
 * @param epmEndpointPackage
 * @param metadataIndex
 * @param policyResponseIndex
 * @param enrollFleet
 * @param generator
 * @param disableEndpointActionsForHost
 */
exports.buildIndexHostsResponse = buildIndexHostsResponse;
const indexEndpointHostDocs = exports.indexEndpointHostDocs = _usage_tracker.usageTracker.track('indexEndpointHostDocs', async ({
  numDocs,
  client,
  kbnClient,
  realPolicies,
  epmEndpointPackage,
  metadataIndex,
  policyResponseIndex,
  enrollFleet,
  generator,
  withResponseActions = true,
  numResponseActions = 1,
  alertIds
}) => {
  const timeBetweenDocs = 6 * 3600 * 1000; // 6 hours between metadata documents
  const timestamp = new Date().getTime();
  const kibanaVersion = await fetchKibanaVersion(kbnClient);
  const activeSpaceId = await (0, _utils.fetchActiveSpaceId)(kbnClient);
  const response = buildIndexHostsResponse();
  response.metadataIndex = metadataIndex;
  response.policyResponseIndex = policyResponseIndex;
  let hostMetadata;
  let wasAgentEnrolled = false;
  const bulkOperations = [];
  for (let j = 0; j < numDocs; j++) {
    generator.updateHostData();
    generator.updateHostPolicyData({
      excludeInitialPolicy: true
    });
    hostMetadata = generator.generateHostMetadata(timestamp - timeBetweenDocs * (numDocs - j - 1), _generate_data.EndpointDocGenerator.createDataStreamFromIndex(metadataIndex));
    let agentId = hostMetadata.agent.id;
    if (enrollFleet) {
      const {
        id: appliedPolicyId,
        name: appliedPolicyName
      } = hostMetadata.Endpoint.policy.applied;
      const uniqueAppliedPolicyName = `${appliedPolicyName}-${(0, _uuid.v4)()}`;

      // If we don't yet have a "real" policy record, then create it now in ingest (package config)
      if (!realPolicies[appliedPolicyId]) {
        const createdPolicies = await (0, _index_fleet_endpoint_policy.indexFleetEndpointPolicy)(kbnClient, uniqueAppliedPolicyName, epmEndpointPackage.version);
        (0, _utils.mergeAndAppendArrays)(response, createdPolicies);

        // eslint-disable-next-line require-atomic-updates
        realPolicies[appliedPolicyId] = createdPolicies.integrationPolicies[0];
      }

      // If we did not yet enroll an agent for this Host, do it now that we have good policy id
      if (!wasAgentEnrolled) {
        var _agentOperations$agen, _agentOperations$agen2, _agentOperations$agen3;
        wasAgentEnrolled = true;
        const agentOperations = {
          agents: [],
          fleetAgentsIndex: '',
          operations: []
        };
        realPolicies[appliedPolicyId].policy_ids.forEach(policyId => {
          const {
            agents,
            fleetAgentsIndex,
            operations
          } = (0, _index_fleet_agent.buildFleetAgentBulkCreateOperations)({
            endpoints: [hostMetadata],
            agentPolicyId: policyId,
            spaceId: activeSpaceId,
            kibanaVersion
          });
          agentOperations.agents = [...agentOperations.agents, ...agents];
          agentOperations.fleetAgentsIndex = fleetAgentsIndex;
          agentOperations.operations = [...agentOperations.operations, ...operations];
        });
        bulkOperations.push(...agentOperations.operations);
        agentId = (_agentOperations$agen = (_agentOperations$agen2 = agentOperations.agents[0]) === null || _agentOperations$agen2 === void 0 ? void 0 : (_agentOperations$agen3 = _agentOperations$agen2.agent) === null || _agentOperations$agen3 === void 0 ? void 0 : _agentOperations$agen3.id) !== null && _agentOperations$agen !== void 0 ? _agentOperations$agen : agentId;
        (0, _utils.mergeAndAppendArrays)(response, {
          agents: agentOperations.agents,
          fleetAgentsIndex: agentOperations.fleetAgentsIndex
        });
      }

      // Update the Host metadata record with the ID of the "real" policy along with the enrolled agent id
      hostMetadata = {
        ...hostMetadata,
        agent: {
          ...hostMetadata.agent,
          id: agentId
        },
        elastic: {
          ...hostMetadata.elastic,
          agent: {
            ...hostMetadata.elastic.agent,
            id: agentId
          }
        },
        Endpoint: {
          ...hostMetadata.Endpoint,
          policy: {
            ...hostMetadata.Endpoint.policy,
            applied: {
              ...hostMetadata.Endpoint.policy.applied,
              id: realPolicies[appliedPolicyId].id
            }
          }
        }
      };

      // Create some fleet endpoint actions and .logs-endpoint actions for this Host
      if (withResponseActions) {
        // `count` logic matches that of `indexEndpointAndFleetActionsForHost()`. Unclear why the number of
        // actions to create will be 5 more than the amount requested if that amount was grater than 1
        const count = numResponseActions === 1 ? numResponseActions : generator.randomN(5) + numResponseActions;
        const {
          operations,
          ...indexFleetActions
        } = (0, _index_endpoint_fleet_actions.buildIEndpointAndFleetActionsBulkOperations)({
          endpoints: [hostMetadata],
          count,
          alertIds
        });
        bulkOperations.push(...operations);
        (0, _utils.mergeAndAppendArrays)(response, indexFleetActions);
      }
    }
    bulkOperations.push({
      create: {
        _index: metadataIndex
      }
    }, hostMetadata);
    const hostPolicyResponse = generator.generatePolicyResponse({
      ts: timestamp - timeBetweenDocs * (numDocs - j - 1),
      policyDataStream: _generate_data.EndpointDocGenerator.createDataStreamFromIndex(policyResponseIndex)
    });
    bulkOperations.push({
      create: {
        _index: policyResponseIndex
      }
    }, hostPolicyResponse);

    // Clone the hostMetadata and policyResponse document to ensure that no shared state
    // (as a result of using the generator) is returned across docs.
    response.hosts.push((0, _lodash.cloneDeep)(hostMetadata));
    response.policyResponses.push((0, _lodash.cloneDeep)(hostPolicyResponse));
  }
  const bulkResponse = await client.bulk({
    operations: bulkOperations,
    refresh: 'wait_for'
  }, {
    headers: {
      'X-elastic-product-origin': 'fleet'
    }
  }).catch(_utils.wrapErrorAndRejectPromise);
  if (bulkResponse.errors) {
    throw new _errors.EndpointError(`indexEndpointHostDocs(): ES Bulk action failed\n\n${JSON.stringify(bulkResponse, null, 2)}`, bulkResponse);
  }
  return response;
});
const fetchKibanaVersion = async kbnClient => {
  const version = (await kbnClient.request({
    path: '/api/status',
    method: 'GET'
  })).data.version.number;
  if (!version) {
    throw new _utils.EndpointDataLoadingError('failed to get kibana version via `/api/status` api');
  }
  return version;
};
const deleteIndexedEndpointHosts = async (esClient, kbnClient, indexedData) => {
  const response = {
    hosts: undefined,
    policyResponses: undefined,
    agents: undefined,
    responses: undefined,
    actions: undefined,
    endpointActionRequests: undefined,
    endpointActionResponses: undefined,
    integrationPolicies: undefined,
    agentPolicies: undefined
  };
  if (indexedData.hosts.length) {
    const query = {
      bool: {
        filter: [{
          terms: {
            'agent.id': indexedData.hosts.map(host => host.agent.id)
          }
        }]
      }
    };
    response.hosts = await esClient.deleteByQuery({
      index: indexedData.metadataIndex,
      wait_for_completion: true,
      query
    }).catch(_utils.wrapErrorAndRejectPromise);

    // Delete from the transform destination index
    await esClient.deleteByQuery({
      index: _constants.metadataCurrentIndexPattern,
      wait_for_completion: true,
      query
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  if (indexedData.policyResponses.length) {
    response.policyResponses = await esClient.deleteByQuery({
      index: indexedData.policyResponseIndex,
      wait_for_completion: true,
      query: {
        bool: {
          filter: [{
            terms: {
              'agent.id': indexedData.policyResponses.map(policyResponse => policyResponse.agent.id)
            }
          }]
        }
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_fleet_agent.deleteIndexedFleetAgents)(esClient, indexedData));
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_endpoint_fleet_actions.deleteIndexedEndpointAndFleetActions)(esClient, indexedData));
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_fleet_endpoint_policy.deleteIndexedFleetEndpointPolicies)(kbnClient, indexedData));
  return response;
};
exports.deleteIndexedEndpointHosts = deleteIndexedEndpointHosts;
/**
 * Indexes a new Endpoint host for a given policy (Endpoint integration policy).
 *
 * NOTE: consider stopping the Endpoint metadata
 */
const indexEndpointHostForPolicy = async ({
  esClient,
  kbnClient,
  integrationPolicyId,
  agentPolicyId,
  overrides = {},
  logger = (0, _utils.createToolingLogger)()
}) => {
  var _integrationPolicy$pa, _integrationPolicy$pa2, _indexedFleetAgent$ag, _indexedFleetAgent$ag2, _integrationPolicy$pa3, _integrationPolicy$pa4;
  const response = buildIndexHostsResponse();
  const [kibanaVersion, integrationPolicy] = await Promise.all([fetchKibanaVersion(kbnClient), kbnClient.request({
    path: _common.packagePolicyRouteService.getInfoPath(integrationPolicyId),
    method: 'GET',
    headers: {
      'elastic-api-version': '2023-10-31'
    }
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow).then(res => res.data.item)]);
  logger.verbose(`Integration policy:\n${JSON.stringify(integrationPolicy, null, 2)}`);
  if (agentPolicyId && !integrationPolicy.policy_ids.includes(agentPolicyId)) {
    throw new _errors.EndpointError(`indexEndpointHostForPolicy(): Invalid agent policy id [${agentPolicyId}]. Agent policy id not listed in integration policy`);
  }
  const agentPolicy = await kbnClient.request({
    method: 'GET',
    path: _common.agentPolicyRouteService.getInfoPath(agentPolicyId !== null && agentPolicyId !== void 0 ? agentPolicyId : integrationPolicy.policy_ids[0]),
    headers: {
      'elastic-api-version': '2023-10-31'
    }
  }).then(res => res.data.item);
  logger.verbose(`Agent policy:\n${JSON.stringify(agentPolicy, null, 2)}`);
  const timestamp = Date.now() - 3.6e6; // Subtract 1 hour

  const docOverrides = (0, _lodash.merge)({
    '@timestamp': timestamp,
    agent: {
      version: kibanaVersion
    },
    Endpoint: {
      policy: {
        applied: {
          name: integrationPolicy.name,
          id: integrationPolicy.id,
          endpoint_policy_version: integrationPolicy.revision,
          status: _types.HostPolicyResponseActionStatus.success
        }
      }
    },
    ...overrides
  });
  const hostMetadataDoc = (0, _lodash.merge)(new _generate_data.EndpointDocGenerator().generateHostMetadata(undefined, _generate_data.EndpointDocGenerator.createDataStreamFromIndex(_constants.METADATA_DATASTREAM)), docOverrides);
  logger.verbose(`New endpoint host metadata doc to be indexed for integration policy [${integrationPolicyId}]:\n${JSON.stringify(hostMetadataDoc, null, 2)}`);
  await (0, _transforms.stopMetadataTransforms)(esClient, (_integrationPolicy$pa = (_integrationPolicy$pa2 = integrationPolicy.package) === null || _integrationPolicy$pa2 === void 0 ? void 0 : _integrationPolicy$pa2.version) !== null && _integrationPolicy$pa !== void 0 ? _integrationPolicy$pa : '');

  // Create the Fleet agent
  const indexedFleetAgent = await (0, _index_fleet_agent.indexFleetAgentForHost)(esClient, hostMetadataDoc, agentPolicyId !== null && agentPolicyId !== void 0 ? agentPolicyId : integrationPolicy.policy_ids[0], kibanaVersion, undefined, agentPolicy.space_ids);
  (0, _utils.mergeAndAppendArrays)(response, indexedFleetAgent);
  logger.info(`New fleet agent indexed [${(_indexedFleetAgent$ag = indexedFleetAgent.agents[0]) === null || _indexedFleetAgent$ag === void 0 ? void 0 : (_indexedFleetAgent$ag2 = _indexedFleetAgent$ag.agent) === null || _indexedFleetAgent$ag2 === void 0 ? void 0 : _indexedFleetAgent$ag2.id}]`);
  logger.verbose(JSON.stringify(indexedFleetAgent.agents, null, 2));
  await esClient.index({
    index: _constants.METADATA_DATASTREAM,
    id: (0, _uuid.v4)(),
    body: hostMetadataDoc,
    op_type: 'create',
    refresh: 'wait_for'
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow);
  response.hosts.push(hostMetadataDoc);
  response.metadataIndex = _constants.METADATA_DATASTREAM;
  logger.info(`New endpoint host metadata doc indexed with agent id [${hostMetadataDoc.agent.id}]`);
  await (0, _transforms.startMetadataTransforms)(esClient, [hostMetadataDoc.agent.id], (_integrationPolicy$pa3 = (_integrationPolicy$pa4 = integrationPolicy.package) === null || _integrationPolicy$pa4 === void 0 ? void 0 : _integrationPolicy$pa4.version) !== null && _integrationPolicy$pa3 !== void 0 ? _integrationPolicy$pa3 : '');
  return response;
};
exports.indexEndpointHostForPolicy = indexEndpointHostForPolicy;