"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VALID_HEX_COLOR = exports.MAX_IMAGE_SIZE = void 0;
exports.createImageHandler = createImageHandler;
exports.getRandomColor = getRandomColor;
exports.readFile = readFile;
exports.resizeImage = resizeImage;
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_IMAGE_SIZE = exports.MAX_IMAGE_SIZE = 64;
function readFile(data) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(data);
  });
}
function resizeImage(imageUrl, maxSize) {
  return new Promise((resolve, reject) => {
    const image = new Image();
    image.onload = () => {
      if (image.width <= maxSize && image.height <= maxSize) {
        return resolve(imageUrl);
      }
      try {
        const canvas = document.createElement('canvas');
        const context = canvas.getContext('2d');
        if (context) {
          if (image.width >= image.height) {
            canvas.width = maxSize;
            canvas.height = Math.floor(image.height * maxSize / image.width);
          } else {
            canvas.height = maxSize;
            canvas.width = Math.floor(image.width * maxSize / image.height);
          }
          context.drawImage(image, 0, 0, canvas.width, canvas.height);
          const resizedDataUrl = canvas.toDataURL();
          return resolve(resizedDataUrl);
        }
      } catch (error) {
        return reject(error);
      }
      return reject();
    };
    image.onerror = reject;
    image.src = imageUrl;
  });
}
function createImageHandler(callback) {
  return async files => {
    if (!files || !files.length) {
      callback(undefined);
      return;
    }
    const file = files[0];
    if (_constants.IMAGE_FILE_TYPES.indexOf(file.type) !== -1) {
      const imageUrl = await readFile(file);
      const resizedImageUrl = await resizeImage(imageUrl, MAX_IMAGE_SIZE);
      callback(resizedImageUrl);
    } else {
      callback(undefined);
    }
  };
}

/**
 * Returns the hex representation of a random color (e.g `#F1B7E2`)
 */
function getRandomColor() {
  return '#' + String(Math.floor(Math.random() * 0xffffff).toString(16)).padStart(6, '0');
}
const VALID_HEX_COLOR = exports.VALID_HEX_COLOR = /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i;