"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineSavedPlaygroundRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _common = require("../../common");
var _types = require("../types");
var _error_handler = require("../utils/error_handler");
var _playgrounds = require("../utils/playgrounds");
var _v = require("../playground_saved_object/schema/v1/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defineSavedPlaygroundRoutes = ({
  logger,
  router
}) => {
  router.versioned.get({
    access: 'internal',
    path: _types.APIRoutes.GET_PLAYGROUNDS,
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    }
  }).addVersion({
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          page: _configSchema.schema.number({
            defaultValue: 1,
            min: 1
          }),
          size: _configSchema.schema.number({
            defaultValue: 10,
            min: 1,
            max: 1000
          }),
          sortField: _configSchema.schema.string({
            defaultValue: 'updated_at'
          }),
          sortOrder: _configSchema.schema.oneOf([_configSchema.schema.literal('desc'), _configSchema.schema.literal('asc')], {
            defaultValue: 'desc'
          })
        })
      }
    },
    version: _common.ROUTE_VERSIONS.v1
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const soClient = (await context.core).savedObjects.client;
    const soPlaygrounds = await soClient.find({
      type: _common.PLAYGROUND_SAVED_OBJECT_TYPE,
      perPage: request.query.size,
      page: request.query.page,
      sortField: request.query.sortField,
      sortOrder: request.query.sortOrder
    });
    const body = (0, _playgrounds.parsePlaygroundSOList)(soPlaygrounds);
    return response.ok({
      body,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.versioned.get({
    access: 'internal',
    path: _types.APIRoutes.GET_PLAYGROUND,
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    }
  }).addVersion({
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      }
    },
    version: _common.ROUTE_VERSIONS.v1
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const soClient = (await context.core).savedObjects.client;
    const soPlayground = await soClient.get(_common.PLAYGROUND_SAVED_OBJECT_TYPE, request.params.id);
    if (soPlayground.error) {
      var _soPlayground$error$m;
      if (soPlayground.error.statusCode === 404) {
        return response.notFound({
          body: {
            message: _i18n.i18n.translate('xpack.searchPlayground.savedPlaygrounds.notFoundError', {
              defaultMessage: '{id} playground not found',
              values: {
                id: request.params.id
              }
            })
          }
        });
      }
      logger.error(_i18n.i18n.translate('xpack.searchPlayground.savedPlaygrounds.getSOError', {
        defaultMessage: 'SavedObject error getting search playground {id}',
        values: {
          id: request.params.id
        }
      }));
      return response.customError({
        statusCode: soPlayground.error.statusCode,
        body: {
          message: soPlayground.error.message,
          attributes: {
            error: soPlayground.error.error,
            ...((_soPlayground$error$m = soPlayground.error.metadata) !== null && _soPlayground$error$m !== void 0 ? _soPlayground$error$m : {})
          }
        }
      });
    }
    const responseBody = (0, _playgrounds.parsePlaygroundSO)(soPlayground);
    return response.ok({
      body: responseBody,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  // Create
  router.versioned.put({
    access: 'internal',
    path: _types.APIRoutes.PUT_PLAYGROUND_CREATE,
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    }
  }).addVersion({
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    version: _common.ROUTE_VERSIONS.v1,
    validate: {
      request: {
        body: _v.playgroundAttributesSchema
      }
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    // Validate playground request
    const playground = request.body;
    const validationErrors = (0, _playgrounds.validatePlayground)(playground);
    if (validationErrors && validationErrors.length > 0) {
      return response.badRequest({
        body: {
          message: _i18n.i18n.translate('xpack.searchPlayground.savedPlaygrounds.validationError', {
            defaultMessage: 'Invalid playground request'
          }),
          attributes: {
            errors: validationErrors
          }
        }
      });
    }
    const soClient = (await context.core).savedObjects.client;
    const soPlayground = await soClient.create(_common.PLAYGROUND_SAVED_OBJECT_TYPE, playground);
    if (soPlayground.error) {
      var _soPlayground$error$m2;
      return response.customError({
        statusCode: soPlayground.error.statusCode,
        body: {
          message: soPlayground.error.message,
          attributes: {
            error: soPlayground.error.error,
            ...((_soPlayground$error$m2 = soPlayground.error.metadata) !== null && _soPlayground$error$m2 !== void 0 ? _soPlayground$error$m2 : {})
          }
        }
      });
    }
    const responseBody = (0, _playgrounds.parsePlaygroundSO)(soPlayground);
    return response.ok({
      body: responseBody,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));

  // Update
  router.versioned.put({
    access: 'internal',
    path: _types.APIRoutes.PUT_PLAYGROUND_UPDATE,
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    }
  }).addVersion({
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    version: _common.ROUTE_VERSIONS.v1,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        }),
        body: _v.playgroundAttributesSchema
      }
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const playground = request.body;
    const validationErrors = (0, _playgrounds.validatePlayground)(playground);
    if (validationErrors && validationErrors.length > 0) {
      return response.badRequest({
        body: {
          message: _i18n.i18n.translate('xpack.searchPlayground.savedPlaygrounds.validationError', {
            defaultMessage: 'Invalid playground request'
          }),
          attributes: {
            errors: validationErrors
          }
        }
      });
    }
    const soClient = (await context.core).savedObjects.client;
    const soPlayground = await soClient.update(_common.PLAYGROUND_SAVED_OBJECT_TYPE, request.params.id, playground);
    if (soPlayground.error) {
      var _soPlayground$error$m3;
      return response.customError({
        statusCode: soPlayground.error.statusCode,
        body: {
          message: soPlayground.error.message,
          attributes: {
            error: soPlayground.error.error,
            ...((_soPlayground$error$m3 = soPlayground.error.metadata) !== null && _soPlayground$error$m3 !== void 0 ? _soPlayground$error$m3 : {})
          }
        }
      });
    }
    return response.ok();
  }));

  // Delete
  router.versioned.delete({
    access: 'internal',
    path: _types.APIRoutes.DELETE_PLAYGROUND,
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    }
  }).addVersion({
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    version: _common.ROUTE_VERSIONS.v1,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string()
        })
      }
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const soClient = (await context.core).savedObjects.client;
    await soClient.delete(_common.PLAYGROUND_SAVED_OBJECT_TYPE, request.params.id);
    return response.ok();
  }));
};
exports.defineSavedPlaygroundRoutes = defineSavedPlaygroundRoutes;