"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deNormalize = void 0;
exports.flattenMappings = flattenMappings;
exports.hasElserOnMlNodeSemanticTextField = hasElserOnMlNodeSemanticTextField;
exports.prepareFieldsForEisUpdate = exports.isElserOnMlNodeSemanticField = void 0;
var _uuid = require("uuid");
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isElserOnMlNodeSemanticField = field => field.source.inference_id === _constants.ELSER_ON_ML_NODE_INFERENCE_ENDPOINT_ID;
exports.isElserOnMlNodeSemanticField = isElserOnMlNodeSemanticField;
function hasElserOnMlNodeSemanticTextField(fields) {
  return Object.values(fields).some(isElserOnMlNodeSemanticField);
}
function flattenMappings(input) {
  const result = {};
  const processMappingNode = (node, name, path, parentId) => {
    var _node$type;
    const id = (0, _uuid.v4)();
    const hasChildren = Boolean(node.properties);
    const entry = {
      id,
      parentId,
      path,
      source: {
        name,
        type: (_node$type = node.type) !== null && _node$type !== void 0 ? _node$type : 'object',
        ...(node.inference_id && {
          inference_id: node.inference_id
        })
      },
      hasChildFields: hasChildren
    };
    if (hasChildren) {
      entry.childFieldsName = 'properties';
      entry.childFields = [];
    }
    result[id] = entry;
    if (node.properties) {
      for (const [childName, childNode] of Object.entries(node.properties)) {
        var _entry$childFields;
        const childId = processMappingNode(childNode, childName, [...path, childName], id);
        (_entry$childFields = entry.childFields) === null || _entry$childFields === void 0 ? void 0 : _entry$childFields.push(childId);
      }
    }
    return id;
  };
  for (const [name, node] of Object.entries(input)) {
    if (isMappingNode(node)) {
      processMappingNode(node, name, [name]);
    }
  }
  return result;
}
function isMappingNode(node) {
  return node.type === 'semantic_text' || typeof node === 'object' && 'properties' in node;
}
const prepareFieldsForEisUpdate = (selectedMappings, flattenedMappings) => {
  const selectedIds = selectedMappings.flatMap(item => {
    var _item$key;
    return (_item$key = item.key) !== null && _item$key !== void 0 ? _item$key : [];
  });
  const resultById = {};
  const resultRootLevel = [];
  function getSelectedFieldData(id) {
    var _clonedField$source;
    // Prevent duplicate processing - if already in resultById, skip
    if (resultById[id]) {
      return;
    }
    const field = flattenedMappings[id];
    if (!field) return;
    const clonedField = {
      ...field
    };

    // Only update inference_id if it exists in source
    if (((_clonedField$source = clonedField.source) === null || _clonedField$source === void 0 ? void 0 : _clonedField$source.inference_id) !== undefined) {
      clonedField.source = {
        ...clonedField.source,
        inference_id: _constants.ELSER_ON_EIS_INFERENCE_ENDPOINT_ID
      };
    }
    resultById[id] = clonedField;

    // Include parent if it exists and hasn't been processed yet
    if (field.parentId) {
      if (!resultById[field.parentId]) {
        getSelectedFieldData(field.parentId);
      }
    } else {
      resultRootLevel.push(id);
    }
  }
  selectedIds.forEach(id => getSelectedFieldData(id));

  // Prune childFields arrays so they only include selected field IDs
  Object.values(resultById).forEach(field => {
    if (field.childFields) {
      field.childFields = field.childFields.filter(id => !!resultById[id]);
      if (field.childFields.length === 0) {
        delete field.childFields;
      }
    }
  });
  return {
    byId: resultById,
    aliases: {},
    rootLevelFields: resultRootLevel
  };
};
exports.prepareFieldsForEisUpdate = prepareFieldsForEisUpdate;
const deNormalize = ({
  rootLevelFields,
  byId
}) => {
  const deNormalizePaths = (ids, to = {}) => {
    ids.forEach(id => {
      const {
        source,
        childFields,
        childFieldsName
      } = byId[id];
      const {
        name,
        ...normalizedField
      } = source;
      const field = normalizedField;
      to[name] = field;
      if (childFields && childFieldsName) {
        field[childFieldsName] = {};
        return deNormalizePaths(childFields, field[childFieldsName]);
      }
    });
    return to;
  };
  return deNormalizePaths(rootLevelFields);
};
exports.deNormalize = deNormalize;