"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchConnectors = exports.fetchConnectorByIndexName = exports.fetchConnectorById = void 0;
var _identify_exceptions = require("../utils/identify_exceptions");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const fetchConnectorById = async (client, connectorId) => {
  try {
    const result = await client.transport.request({
      method: 'GET',
      path: `/_connector/${connectorId}`
    });
    return result;
  } catch (err) {
    if ((0, _identify_exceptions.isNotFoundException)(err)) {
      return undefined;
    }
    throw err;
  }
};
exports.fetchConnectorById = fetchConnectorById;
const fetchConnectorByIndexName = async (client, indexName) => {
  try {
    const connectorListResult = await client.transport.request({
      method: 'GET',
      path: `/_connector`,
      querystring: {
        index_name: indexName
      }
    });
    const result = connectorListResult.count > 0 ? connectorListResult.results[0] : undefined;
    return result;
  } catch (error) {
    throw error;
  }
};
exports.fetchConnectorByIndexName = fetchConnectorByIndexName;
const fetchConnectors = async (client, indexNames, fetchOnlyCrawlers, searchQuery, includeDeleted) => {
  const q = searchQuery && searchQuery.length > 0 ? searchQuery : undefined;
  const querystring = q ? {
    query: q
  } : indexNames ? {
    index_name: indexNames.join(',')
  } : {};
  let hits = [];
  let accumulator = [];
  do {
    const connectorResult = await client.transport.request({
      method: 'GET',
      path: `/_connector`,
      querystring: {
        ...querystring,
        from: accumulator.length,
        size: 1000,
        include_deleted: includeDeleted
      }
    });
    hits = connectorResult.results;
    accumulator = accumulator.concat(hits);
  } while (hits.length >= 1000);
  const result = accumulator;
  if (fetchOnlyCrawlers !== undefined) {
    return result.filter(hit => {
      return !fetchOnlyCrawlers ? hit.service_type !== _.CRAWLER_SERVICE_TYPE : hit.service_type === _.CRAWLER_SERVICE_TYPE;
    });
  }
  return result;
};
exports.fetchConnectors = fetchConnectors;