"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTotalCountAggregations = getTotalCountAggregations;
exports.parseJobTypeBucket = parseJobTypeBucket;
var _parse_and_log_error = require("./parse_and_log_error");
var _saved_objects = require("../../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTotalCountAggregations({
  esClient,
  index,
  logger
}) {
  try {
    var _results$hits$total, _aggregations$enabled, _aggregations$has_not;
    const query = {
      index,
      size: 0,
      track_total_hits: true,
      query: {
        bool: {
          // Aggregate over all scheduled report saved objects
          filter: [{
            term: {
              type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
            }
          }]
        }
      },
      runtime_mappings: {
        'scheduled_report.enabled': {
          type: 'boolean'
        },
        'scheduled_report.jobType': {
          type: 'keyword'
        },
        'scheduled_report.notification.email.to': {
          type: 'keyword'
        }
      },
      aggs: {
        by_job_type: {
          terms: {
            field: 'scheduled_report.jobType',
            size: 20
          }
        },
        enabled: {
          filter: {
            term: {
              'scheduled_report.enabled': true
            }
          },
          aggs: {
            by_job_type: {
              terms: {
                field: 'scheduled_report.jobType',
                size: 20
              }
            }
          }
        },
        has_notifications: {
          filter: {
            exists: {
              field: 'scheduled_report.notification.email.to'
            }
          }
        }
      }
    };
    logger.debug(() => `query for getTotalCountAggregations - ${JSON.stringify(query)}`);
    const results = await esClient.search(query);
    logger.debug(() => `results for getTotalCountAggregations query - ${JSON.stringify(results)}`);
    const aggregations = results.aggregations;
    const totalReportsCount = typeof results.hits.total === 'number' ? results.hits.total : (_results$hits$total = results.hits.total) === null || _results$hits$total === void 0 ? void 0 : _results$hits$total.value;
    const countReportsByJobType = parseJobTypeBucket(aggregations.by_job_type.buckets);
    const countEnabledReportsByJobType = parseJobTypeBucket(aggregations.enabled.by_job_type.buckets);
    return {
      hasErrors: false,
      number_of_scheduled_reports: totalReportsCount !== null && totalReportsCount !== void 0 ? totalReportsCount : 0,
      number_of_enabled_scheduled_reports: (_aggregations$enabled = aggregations.enabled.doc_count) !== null && _aggregations$enabled !== void 0 ? _aggregations$enabled : 0,
      number_of_scheduled_reports_by_type: countReportsByJobType,
      number_of_enabled_scheduled_reports_by_type: countEnabledReportsByJobType,
      number_of_scheduled_reports_with_notifications: (_aggregations$has_not = aggregations.has_notifications.doc_count) !== null && _aggregations$has_not !== void 0 ? _aggregations$has_not : 0
    };
  } catch (err) {
    const errorMessage = (0, _parse_and_log_error.parseAndLogError)(err, `getTotalCountAggregations`, logger);
    return {
      hasErrors: true,
      errorMessage,
      number_of_scheduled_reports: 0,
      number_of_enabled_scheduled_reports: 0,
      number_of_scheduled_reports_by_type: {},
      number_of_enabled_scheduled_reports_by_type: {},
      number_of_scheduled_reports_with_notifications: 0
    };
  }
}
function parseJobTypeBucket(jobTypeBuckets) {
  const buckets = jobTypeBuckets;
  return (buckets !== null && buckets !== void 0 ? buckets : []).reduce((acc, bucket) => {
    var _bucket$doc_count;
    const jobType = `${bucket.key}`;
    acc[jobType] = (_bucket$doc_count = bucket.doc_count) !== null && _bucket$doc_count !== void 0 ? _bucket$doc_count : 0;
    return acc;
  }, {});
}