"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScheduledReportsService = void 0;
var _reportingServer = require("@kbn/reporting-server");
var _lodash = require("lodash");
var _saved_objects = require("../../saved_objects");
var _audit_events = require("../audit_events/audit_events");
var _constants = require("./constants");
var _transforms = require("./transforms");
var _update = require("./schemas/update");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SCHEDULED_REPORT_ID_FIELD = 'scheduled_report_id';
const CREATED_AT_FIELD = 'created_at';
class ScheduledReportsService {
  constructor(auditLogger, userCanManageReporting, esClient, logger, responseFactory, savedObjectsClient, taskManager, request) {
    this.auditLogger = auditLogger;
    this.userCanManageReporting = userCanManageReporting;
    this.esClient = esClient;
    this.logger = logger;
    this.responseFactory = responseFactory;
    this.savedObjectsClient = savedObjectsClient;
    this.taskManager = taskManager;
    this.request = request;
  }
  static async build({
    logger,
    reportingCore,
    responseFactory,
    request
  }) {
    const esClient = await reportingCore.getEsClient();
    const auditLogger = await reportingCore.getAuditLogger(request);
    const savedObjectsClient = await reportingCore.getScopedSoClient(request);
    const taskManager = await reportingCore.getTaskManager();
    const userCanManageReporting = await reportingCore.canManageReportingForSpace(request);
    return new ScheduledReportsService(auditLogger, userCanManageReporting, esClient, logger, responseFactory, savedObjectsClient, taskManager, request);
  }
  async update({
    user,
    id,
    updateParams
  }) {
    try {
      _update.updateScheduledReportSchema.validate(updateParams);
    } catch (error) {
      throw this.responseFactory.badRequest({
        body: `Error validating params for update scheduled report - ${error.message}`
      });
    }
    if (!(await this._canUpdateReport({
      id,
      user
    }))) {
      this._throw404({
        user,
        id,
        action: _audit_events.ScheduledReportAuditAction.UPDATE
      });
    }
    try {
      const {
        title,
        schedule,
        notification
      } = updateParams;
      await this._updateScheduledReportSavedObject({
        id,
        title,
        schedule,
        notification
      });
      await this._updateScheduledReportTaskSchedule({
        id,
        schedule
      });
      const updatedReport = await this.savedObjectsClient.get(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, id);
      this._auditLog({
        action: _audit_events.ScheduledReportAuditAction.UPDATE,
        id,
        name: updatedReport.attributes.title
      });
      return (0, _transforms.transformSingleResponse)(this.logger, updatedReport);
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error updating scheduled reports: ${error.message}`
      });
    }
  }
  async list({
    user,
    page = 1,
    size = _constants.DEFAULT_SCHEDULED_REPORT_LIST_SIZE,
    search
  }) {
    try {
      const username = this._getUsername(user);
      const response = await this.savedObjectsClient.find({
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
        page,
        perPage: size,
        search,
        searchFields: ['title', 'created_by'],
        ...(!this.userCanManageReporting ? {
          filter: `scheduled_report.attributes.createdBy: "${username}"`
        } : {})
      });
      if (!response) {
        return this._getEmptyListApiResponse(page, size);
      }
      const scheduledReportIdsAndName = response === null || response === void 0 ? void 0 : response.saved_objects.map(so => ({
        id: so.id,
        name: so.attributes.title
      }));
      if (!scheduledReportIdsAndName || scheduledReportIdsAndName.length === 0) {
        return this._getEmptyListApiResponse(page, size);
      }
      scheduledReportIdsAndName.forEach(({
        id,
        name
      }) => this._auditLog({
        action: _audit_events.ScheduledReportAuditAction.LIST,
        id,
        name
      }));
      const scheduledReportIds = scheduledReportIdsAndName.map(({
        id
      }) => id);
      let lastRunResponse;
      try {
        lastRunResponse = await this.esClient.asInternalUser.search({
          index: _reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY,
          size,
          _source: [CREATED_AT_FIELD],
          sort: [{
            [CREATED_AT_FIELD]: {
              order: 'desc'
            }
          }],
          query: {
            bool: {
              filter: [{
                terms: {
                  [SCHEDULED_REPORT_ID_FIELD]: scheduledReportIds
                }
              }]
            }
          },
          collapse: {
            field: SCHEDULED_REPORT_ID_FIELD
          }
        });
      } catch (error) {
        // if no scheduled reports have run yet, we will get an error from the collapse query
        // ignore these and return an empty last run
        this.logger.warn(`Error getting last run for scheduled reports: ${error.message}`);
      }
      let nextRunResponse;
      try {
        nextRunResponse = await this.taskManager.bulkGet(scheduledReportIds);
      } catch (error) {
        // swallow this error
        this.logger.warn(`Error getting next run for scheduled reports: ${error.message}`);
      }
      return (0, _transforms.transformListResponse)(this.logger, response, lastRunResponse, nextRunResponse);
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error listing scheduled reports: ${error.message}`
      });
    }
  }
  async bulkDisable({
    ids,
    user
  }) {
    return this._bulkOperation({
      enable: false,
      ids,
      user
    });
  }
  async bulkEnable({
    ids,
    user
  }) {
    return this._bulkOperation({
      enable: true,
      ids,
      user
    });
  }
  async bulkDelete({
    ids,
    user
  }) {
    try {
      const username = this._getUsername(user);
      const bulkGetResult = await this.savedObjectsClient.bulkGet(ids.map(id => ({
        id,
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
      })));
      const [validSchedules, bulkGetErrors] = (0, _lodash.partition)(bulkGetResult.saved_objects, so => so.error === undefined);
      const [authorizedSchedules, unauthorizedSchedules] = (0, _lodash.partition)(validSchedules, so => so.attributes.createdBy === username || this.userCanManageReporting);
      const authErrors = this._formatAndAuditBulkDeleteAuthErrors({
        bulkGetErrors,
        unauthorizedSchedules,
        username
      });
      this._auditBulkGetAuthorized({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        authorizedSchedules
      });
      if (authorizedSchedules.length === 0) {
        return (0, _transforms.transformBulkDeleteResponse)({
          deletedSchedulesIds: [],
          errors: authErrors
        });
      }
      const bulkDeleteResult = await this.savedObjectsClient.bulkDelete(authorizedSchedules.map(so => ({
        id: so.id,
        type: so.type
      })));
      const [deletedSchedules, bulkDeleteErrors] = (0, _lodash.partition)(bulkDeleteResult.statuses, status => status.error === undefined);
      const executionErrors = this._formatAndAuditBulkDeleteSchedulesErrors({
        errorStatuses: bulkDeleteErrors
      });
      const removeTasksResult = await this.taskManager.bulkRemove(deletedSchedules.map(so => so.id));
      const [removedTasks, erroredTasks] = (0, _lodash.partition)(removeTasksResult.statuses, status => Boolean(status.success));
      const taskErrors = this._formatBulkDeleteTaskErrors({
        errorStatuses: erroredTasks
      });
      return (0, _transforms.transformBulkDeleteResponse)({
        deletedSchedulesIds: removedTasks.map(task => task.id),
        errors: [...authErrors, ...executionErrors, ...taskErrors]
      });
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error deleting scheduled reports: ${error.message}`
      });
    }
  }
  _auditBulkGetAuthorized({
    action,
    authorizedSchedules
  }) {
    authorizedSchedules.forEach(so => {
      this._auditLog({
        action,
        id: so.id,
        name: so.attributes.title,
        outcome: 'unknown'
      });
    });
  }
  _formatAndAuditBulkDeleteAuthErrors({
    bulkGetErrors,
    unauthorizedSchedules,
    username
  }) {
    const bulkErrors = [];
    bulkGetErrors.forEach(so => {
      if (!so.error) {
        return;
      }
      bulkErrors.push({
        message: so.error.message,
        status: so.error.statusCode,
        id: so.id
      });
    });
    unauthorizedSchedules.forEach(so => {
      var _so$attributes;
      bulkErrors.push({
        message: `Not found.`,
        status: 404,
        id: so.id
      });
      this.logger.warn(`User "${username}" attempted to delete scheduled report "${so.id}" created by "${so.attributes.createdBy}" without sufficient privileges.`);
      this._auditLog({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        id: so.id,
        name: so === null || so === void 0 ? void 0 : (_so$attributes = so.attributes) === null || _so$attributes === void 0 ? void 0 : _so$attributes.title,
        outcome: 'failure',
        error: new Error(`Not found.`)
      });
    });
    return bulkErrors;
  }
  _formatAndAuditBulkDeleteSchedulesErrors({
    errorStatuses
  }) {
    const bulkErrors = [];
    errorStatuses.forEach(status => {
      if (!status.error) {
        return;
      }
      bulkErrors.push({
        message: status.error.message,
        status: status.error.statusCode,
        id: status.id
      });
      this._auditLog({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        id: status.id,
        error: new Error(status.error.message)
      });
    });
    return bulkErrors;
  }
  _formatBulkDeleteTaskErrors({
    errorStatuses
  }) {
    const bulkErrors = [];
    errorStatuses.forEach(error => {
      if (error.error == null) {
        return;
      }
      bulkErrors.push({
        message: `Scheduled report deleted but task deleting failed due to: ${error.error.message}`,
        status: error.error.statusCode,
        id: error.id
      });
    });
    return bulkErrors;
  }
  _getUsername(user) {
    return user ? user.username : false;
  }
  _getEmptyListApiResponse(page, perPage) {
    return {
      page,
      per_page: perPage,
      total: 0,
      data: []
    };
  }
  _auditLog({
    action,
    id,
    name,
    outcome,
    error
  }) {
    this.auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
      action,
      savedObject: {
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
        id,
        name
      },
      outcome,
      error
    }));
  }
  async _updateScheduledReportSavedObject({
    id,
    title,
    schedule,
    notification
  }) {
    await this.savedObjectsClient.update(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, id, {
      title,
      schedule,
      notification
    });
  }
  async _updateScheduledReportTaskSchedule({
    id,
    schedule
  }) {
    if (schedule) {
      await this.taskManager.bulkUpdateSchedules([id], schedule, {
        request: this.request
      });
    }
  }
  async _canUpdateReport({
    user,
    id
  }) {
    if (this.userCanManageReporting) return true;
    const username = this._getUsername(user);
    const reportToUpdate = await this.savedObjectsClient.get(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, id);
    return reportToUpdate.attributes.createdBy === username;
  }
  async _bulkOperation({
    enable,
    ids,
    user
  }) {
    try {
      let taskIdsToUpdate = [];
      const bulkGetResult = await this.savedObjectsClient.bulkGet(ids.map(id => ({
        id,
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
      })));
      const {
        errors: bulkErrors,
        scheduledReportSavedObjectsToUpdate,
        updatedScheduledReportIds: enabledScheduledReportIds
      } = await this._addLogForBulkOperationScheduledReports({
        action: enable ? _audit_events.ScheduledReportAuditAction.ENABLE : _audit_events.ScheduledReportAuditAction.DISABLE,
        scheduledReportSavedObjects: bulkGetResult.saved_objects,
        user,
        operation: enable ? 'enable' : 'disable'
      });

      // nothing to update, return early
      if (scheduledReportSavedObjectsToUpdate.length > 0) {
        const bulkUpdateResult = await this._updateScheduledReportSavedObjectEnabledState({
          scheduledReportSavedObjectsToUpdate,
          shouldEnable: enable
        });
        for (const so of bulkUpdateResult.saved_objects) {
          if (so.error) {
            var _so$attributes2;
            bulkErrors.push({
              message: so.error.message,
              status: so.error.statusCode,
              id: so.id
            });
            this._auditLog({
              action: enable ? _audit_events.ScheduledReportAuditAction.ENABLE : _audit_events.ScheduledReportAuditAction.DISABLE,
              id: so.id,
              name: so === null || so === void 0 ? void 0 : (_so$attributes2 = so.attributes) === null || _so$attributes2 === void 0 ? void 0 : _so$attributes2.title,
              error: new Error(so.error.message)
            });
          } else {
            taskIdsToUpdate.push(so.id);
          }
        }
      } else {
        return {
          scheduled_report_ids: [...enabledScheduledReportIds],
          errors: bulkErrors,
          total: enabledScheduledReportIds.size + bulkErrors.length
        };
      }
      taskIdsToUpdate = taskIdsToUpdate.concat([...enabledScheduledReportIds]);
      return this._updateScheduledReportTaskEnabledState({
        taskIdsToUpdate,
        shouldEnable: enable,
        bulkErrors,
        updatedScheduledReportIds: enabledScheduledReportIds
      });
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error ${enable ? 'enabling' : 'disabling'} scheduled reports: ${error.message}`
      });
    }
  }
  async _updateScheduledReportSavedObjectEnabledState({
    scheduledReportSavedObjectsToUpdate,
    shouldEnable
  }) {
    return await this.savedObjectsClient.bulkUpdate(scheduledReportSavedObjectsToUpdate.map(so => ({
      id: so.id,
      type: so.type,
      attributes: {
        enabled: shouldEnable
      }
    })));
  }
  async _addLogForBulkOperationScheduledReports({
    action,
    scheduledReportSavedObjects,
    user,
    operation
  }) {
    const errors = [];
    const scheduledReportSavedObjectsToUpdate = [];
    const username = this._getUsername(user);
    const updatedScheduledReportIds = new Set();
    for (const so of scheduledReportSavedObjects) {
      if (so.error) {
        errors.push({
          message: so.error.message,
          status: so.error.statusCode,
          id: so.id
        });
      } else {
        // check if user is allowed to update this scheduled report
        if (so.attributes.createdBy !== username && !this.userCanManageReporting) {
          var _so$attributes3;
          errors.push({
            message: `Not found.`,
            status: 404,
            id: so.id
          });
          this.logger.warn(`User "${username}" attempted to ${operation} scheduled report "${so.id}" created by "${so.attributes.createdBy}" without sufficient privileges.`);
          this._auditLog({
            action,
            id: so.id,
            name: so === null || so === void 0 ? void 0 : (_so$attributes3 = so.attributes) === null || _so$attributes3 === void 0 ? void 0 : _so$attributes3.title,
            error: new Error('Not found.')
          });
        } else if (operation === 'disable' && so.attributes.enabled === false) {
          this.logger.debug(`Scheduled report ${so.id} is already disabled`);
          updatedScheduledReportIds.add(so.id);
        } else if (operation === 'enable' && so.attributes.enabled === true) {
          this.logger.debug(`Scheduled report ${so.id} is already enabled`);
          updatedScheduledReportIds.add(so.id);
        } else {
          this._auditLog({
            action,
            id: so.id,
            name: so.attributes.title,
            outcome: 'unknown'
          });
          scheduledReportSavedObjectsToUpdate.push(so);
        }
      }
    }
    return {
      errors,
      scheduledReportSavedObjectsToUpdate,
      updatedScheduledReportIds
    };
  }
  async _updateScheduledReportTaskEnabledState({
    taskIdsToUpdate,
    shouldEnable,
    bulkErrors,
    updatedScheduledReportIds
  }) {
    const resultFromUpdatingTasks = shouldEnable ? await this.taskManager.bulkEnable(taskIdsToUpdate, false, {
      request: this.request
    }) : await this.taskManager.bulkDisable(taskIdsToUpdate, false, {
      request: this.request
    });
    for (const error of resultFromUpdatingTasks.errors) {
      bulkErrors.push({
        message: `Scheduled report ${shouldEnable ? 'enabled' : 'disabled'} but task ${shouldEnable ? 'enabling' : 'disabling'} failed due to: ${error.error.message}`,
        status: error.error.statusCode,
        id: error.id
      });
    }
    for (const result of resultFromUpdatingTasks.tasks) {
      updatedScheduledReportIds.add(result.id);
    }
    return {
      scheduled_report_ids: [...updatedScheduledReportIds],
      errors: bulkErrors,
      total: updatedScheduledReportIds.size + bulkErrors.length
    };
  }
  _throw404({
    user,
    id,
    action
  }) {
    const username = this._getUsername(user);
    this.logger.warn(`User "${username}" attempted to update scheduled report "${id}" without sufficient privileges.`);
    this._auditLog({
      action,
      id,
      error: new Error('Not found.')
    });
    throw this.responseFactory.customError({
      statusCode: 404,
      body: 'Not found.'
    });
  }
}
exports.ScheduledReportsService = ScheduledReportsService;