"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PackageInstaller = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _productDocCommon = require("@kbn/product-doc-common");
var _inferenceCommon = require("@kbn/inference-common");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _is_default_inference_endpoint = require("@kbn/product-doc-common/src/is_default_inference_endpoint");
var _utils = require("./utils");
var _semver = require("./utils/semver");
var _steps = require("./steps");
var _create_index = require("./steps/create_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PackageInstaller {
  constructor({
    artifactsFolder,
    logger,
    esClient,
    productDocClient,
    artifactRepositoryUrl,
    elserInferenceId,
    kibanaVersion
  }) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "artifactsFolder", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "productDocClient", void 0);
    (0, _defineProperty2.default)(this, "artifactRepositoryUrl", void 0);
    (0, _defineProperty2.default)(this, "currentVersion", void 0);
    (0, _defineProperty2.default)(this, "elserInferenceId", void 0);
    this.esClient = esClient;
    this.productDocClient = productDocClient;
    this.artifactsFolder = artifactsFolder;
    this.artifactRepositoryUrl = artifactRepositoryUrl;
    this.currentVersion = (0, _semver.majorMinor)(kibanaVersion);
    this.log = logger;
    this.elserInferenceId = elserInferenceId || _inferenceCommon.defaultInferenceEndpoints.ELSER;
  }
  async getInferenceInfo(inferenceId) {
    if (!inferenceId) {
      return;
    }
    const inferenceEndpoints = await this.esClient.inference.get({
      inference_id: inferenceId
    });
    return Array.isArray(inferenceEndpoints.endpoints) && inferenceEndpoints.endpoints.length > 0 ? inferenceEndpoints.endpoints[0] : undefined;
  }
  /**
   * Make sure that the currently installed doc packages are up to date.
   * Will not upgrade products that are not already installed
   */
  async ensureUpToDate(params) {
    const {
      inferenceId,
      forceUpdate
    } = params;
    const inferenceInfo = await this.getInferenceInfo(inferenceId);
    const [repositoryVersions, installStatuses] = await Promise.all([(0, _steps.fetchArtifactVersions)({
      artifactRepositoryUrl: this.artifactRepositoryUrl
    }), this.productDocClient.getInstallationStatus({
      inferenceId
    })]);
    const toUpdate = [];
    Object.entries(installStatuses).forEach(([productName, productState]) => {
      if (productState.status === 'uninstalled') {
        return;
      }
      const availableVersions = repositoryVersions[productName];
      if (!availableVersions || !availableVersions.length) {
        return;
      }
      const selectedVersion = selectVersion(this.currentVersion, availableVersions);
      if (productState.version !== selectedVersion || Boolean(forceUpdate)) {
        toUpdate.push({
          productName: productName,
          productVersion: selectedVersion
        });
      }
    });
    for (const {
      productName,
      productVersion
    } of toUpdate) {
      await this.installPackage({
        productName,
        productVersion,
        customInference: inferenceInfo
      });
    }
  }
  async installAll(params = {}) {
    const {
      inferenceId
    } = params;
    const repositoryVersions = await (0, _steps.fetchArtifactVersions)({
      artifactRepositoryUrl: this.artifactRepositoryUrl
    });
    const allProducts = Object.values(_productDocCommon.DocumentationProduct);
    const inferenceInfo = await this.getInferenceInfo(inferenceId);
    for (const productName of allProducts) {
      const availableVersions = repositoryVersions[productName];
      if (!availableVersions || !availableVersions.length) {
        this.log.warn(`No version found for product [${productName}]`);
        continue;
      }
      const selectedVersion = selectVersion(this.currentVersion, availableVersions);
      await this.installPackage({
        productName,
        productVersion: selectedVersion,
        customInference: inferenceInfo
      });
    }
  }
  async installPackage({
    productName,
    productVersion,
    customInference
  }) {
    var _customInference$infe;
    const inferenceId = (_customInference$infe = customInference === null || customInference === void 0 ? void 0 : customInference.inference_id) !== null && _customInference$infe !== void 0 ? _customInference$infe : this.elserInferenceId;
    this.log.info(`Starting installing documentation for product [${productName}] and version [${productVersion}] with inference ID [${inferenceId}]`);
    productVersion = (0, _semver.majorMinor)(productVersion);
    await this.uninstallPackage({
      productName,
      inferenceId
    });
    let zipArchive;
    try {
      var _customInference$infe2;
      await this.productDocClient.setInstallationStarted({
        productName,
        productVersion,
        inferenceId
      });
      if (customInference && !(0, _is_default_inference_endpoint.isImpliedDefaultElserInferenceId)(customInference === null || customInference === void 0 ? void 0 : customInference.inference_id)) {
        if ((customInference === null || customInference === void 0 ? void 0 : customInference.task_type) !== 'text_embedding') {
          throw new Error(`Inference [${inferenceId}]'s task type ${customInference === null || customInference === void 0 ? void 0 : customInference.task_type} is not supported. Please use a model with task type 'text_embedding'.`);
        }
        await (0, _utils.ensureInferenceDeployed)({
          client: this.esClient,
          inferenceId
        });
      }
      if (!customInference || (0, _is_default_inference_endpoint.isImpliedDefaultElserInferenceId)(customInference === null || customInference === void 0 ? void 0 : customInference.inference_id)) {
        await (0, _utils.ensureDefaultElserDeployed)({
          client: this.esClient
        });
      }
      const artifactFileName = (0, _productDocCommon.getArtifactName)({
        productName,
        productVersion,
        inferenceId: (_customInference$infe2 = customInference === null || customInference === void 0 ? void 0 : customInference.inference_id) !== null && _customInference$infe2 !== void 0 ? _customInference$infe2 : this.elserInferenceId
      });
      const artifactUrl = `${this.artifactRepositoryUrl}/${artifactFileName}`;
      const artifactPathAtVolume = `${this.artifactsFolder}/${artifactFileName}`;
      this.log.debug(`Downloading from [${artifactUrl}] to [${artifactPathAtVolume}]`);
      const artifactFullPath = await (0, _utils.downloadToDisk)(artifactUrl, artifactPathAtVolume);
      zipArchive = await (0, _utils.openZipArchive)(artifactFullPath);
      (0, _steps.validateArtifactArchive)(zipArchive);
      const [manifest, mappings] = await Promise.all([(0, _utils.loadManifestFile)(zipArchive), (0, _utils.loadMappingFile)(zipArchive)]);
      const manifestVersion = manifest.formatVersion;
      const indexName = (0, _productDocCommon.getProductDocIndexName)(productName, customInference === null || customInference === void 0 ? void 0 : customInference.inference_id);
      const modifiedMappings = (0, _lodash.cloneDeep)(mappings);
      (0, _create_index.overrideInferenceSettings)(modifiedMappings, inferenceId);
      await (0, _steps.createIndex)({
        indexName,
        mappings: modifiedMappings,
        // Mappings will be overridden by the inference ID and inference type
        manifestVersion,
        esClient: this.esClient,
        log: this.log
      });
      await (0, _steps.populateIndex)({
        indexName,
        manifestVersion,
        archive: zipArchive,
        esClient: this.esClient,
        log: this.log,
        inferenceId
      });
      await this.productDocClient.setInstallationSuccessful(productName, indexName, inferenceId);
      this.log.info(`Documentation installation successful for product [${productName}] and version [${productVersion}]`);
    } catch (e) {
      let message = e.message;
      if (message.includes('End of central directory record signature not found.')) {
        message = _i18n.i18n.translate('aiInfra.productDocBase.packageInstaller.noArtifactAvailable', {
          values: {
            productName,
            productVersion,
            inferenceId
          },
          defaultMessage: 'No documentation artifact available for product [{productName}]/[{productVersion}] for Inference ID [{inferenceId}]. Please select a different model or contact your administrator.'
        });
      }
      this.log.error(`Error during documentation installation of product [${productName}]/[${productVersion}] : ${message}`);
      await this.productDocClient.setInstallationFailed(productName, message, inferenceId);
      throw e;
    } finally {
      var _zipArchive;
      (_zipArchive = zipArchive) === null || _zipArchive === void 0 ? void 0 : _zipArchive.close();
    }
  }
  async uninstallPackage({
    productName,
    inferenceId
  }) {
    const indexName = (0, _productDocCommon.getProductDocIndexName)(productName, inferenceId);
    await this.esClient.indices.delete({
      index: indexName
    }, {
      ignore: [404]
    });
    await this.productDocClient.setUninstalled(productName, inferenceId);
  }
  async uninstallAll(params = {}) {
    const {
      inferenceId
    } = params;
    const allProducts = Object.values(_productDocCommon.DocumentationProduct);
    for (const productName of allProducts) {
      await this.productDocClient.setUninstallationStarted(productName, inferenceId);
      await this.uninstallPackage({
        productName,
        inferenceId
      });
    }
  }

  // Security Labs methods

  /**
   * Install Security Labs content from the CDN.
   */
  async installSecurityLabs({
    version,
    inferenceId
  }) {
    const effectiveInferenceId = inferenceId || this.elserInferenceId;
    this.log.info(`Starting Security Labs installation${version ? ` for version [${version}]` : ''} with inference ID [${effectiveInferenceId}]`);

    // Uninstall existing Security Labs content first
    await this.uninstallSecurityLabs({
      inferenceId: effectiveInferenceId
    });
    let zipArchive;
    let selectedVersion;
    try {
      // Ensure ELSER is deployed
      await (0, _utils.ensureDefaultElserDeployed)({
        client: this.esClient
      });

      // Determine version to install
      selectedVersion = version;
      if (!selectedVersion) {
        const availableVersions = await (0, _steps.fetchSecurityLabsVersions)({
          artifactRepositoryUrl: this.artifactRepositoryUrl
        });
        if (availableVersions.length === 0) {
          throw new Error('No Security Labs versions available');
        }
        // Select the latest version
        selectedVersion = availableVersions.sort().reverse()[0];
      }
      await this.productDocClient.setSecurityLabsInstallationStarted({
        version: selectedVersion,
        inferenceId: effectiveInferenceId
      });
      const artifactFileName = (0, _productDocCommon.getSecurityLabsArtifactName)({
        version: selectedVersion,
        inferenceId: effectiveInferenceId
      });
      const artifactUrl = `${this.artifactRepositoryUrl}/${artifactFileName}`;
      const artifactPath = `${this.artifactsFolder}/${artifactFileName}`;
      this.log.debug(`Downloading Security Labs from [${artifactUrl}] to [${artifactPath}]`);
      const downloadedFullPath = await (0, _utils.downloadToDisk)(artifactUrl, artifactPath);
      zipArchive = await (0, _utils.openZipArchive)(downloadedFullPath);
      (0, _steps.validateArtifactArchive)(zipArchive);
      const [manifest, mappings] = await Promise.all([(0, _utils.loadManifestFile)(zipArchive), (0, _utils.loadMappingFile)(zipArchive)]);
      const manifestVersion = manifest.formatVersion;
      const indexName = (0, _productDocCommon.getSecurityLabsIndexName)(effectiveInferenceId);
      const modifiedMappings = (0, _lodash.cloneDeep)(mappings);
      (0, _create_index.overrideInferenceSettings)(modifiedMappings, effectiveInferenceId);
      await (0, _steps.createIndex)({
        indexName,
        mappings: modifiedMappings,
        manifestVersion,
        esClient: this.esClient,
        log: this.log
      });
      await (0, _steps.populateIndex)({
        indexName,
        manifestVersion,
        archive: zipArchive,
        esClient: this.esClient,
        log: this.log,
        inferenceId: effectiveInferenceId
      });
      await this.productDocClient.setSecurityLabsInstallationSuccessful({
        version: selectedVersion,
        indexName,
        inferenceId: effectiveInferenceId
      });
      this.log.info(`Security Labs installation successful for version [${selectedVersion}]`);
    } catch (e) {
      let message = e.message;
      if (message.includes('End of central directory record signature not found.')) {
        message = _i18n.i18n.translate('aiInfra.productDocBase.packageInstaller.noSecurityLabsArtifactAvailable', {
          values: {
            inferenceId: effectiveInferenceId
          },
          defaultMessage: 'No Security Labs artifact available for Inference ID [{inferenceId}]. Please contact your administrator.'
        });
      }
      this.log.error(`Error during Security Labs installation: ${message}`);
      await this.productDocClient.setSecurityLabsInstallationFailed({
        version: selectedVersion,
        failureReason: message,
        inferenceId: effectiveInferenceId
      });
      throw e;
    } finally {
      var _zipArchive2;
      (_zipArchive2 = zipArchive) === null || _zipArchive2 === void 0 ? void 0 : _zipArchive2.close();
    }
  }

  /**
   * Uninstall Security Labs content.
   */
  async uninstallSecurityLabs({
    inferenceId
  }) {
    const indexName = (0, _productDocCommon.getSecurityLabsIndexName)(inferenceId);
    await this.esClient.indices.delete({
      index: indexName
    }, {
      ignore: [404]
    });
    if (inferenceId) {
      await this.productDocClient.setSecurityLabsUninstalled(inferenceId);
    }
    this.log.info(`Security Labs content uninstalled from index [${indexName}]`);
  }

  /**
   * Get the installation status of Security Labs content.
   */
  async getSecurityLabsStatus({
    inferenceId
  }) {
    try {
      const effectiveInferenceId = inferenceId !== null && inferenceId !== void 0 ? inferenceId : this.elserInferenceId;
      const status = await this.productDocClient.getSecurityLabsInstallationStatus({
        inferenceId: effectiveInferenceId
      });

      // Compute latest version (best-effort) for UX and auto-update checks.
      let repoLatestVersion;
      try {
        const versions = await (0, _steps.fetchSecurityLabsVersions)({
          artifactRepositoryUrl: this.artifactRepositoryUrl
        });
        if (versions.length > 0) {
          repoLatestVersion = versions.slice().sort().reverse()[0];
        }
      } catch (e) {
        // ignore
      }
      const installedVersion = status.version;
      const isUpdateAvailable = status.status === 'installed' && Boolean(installedVersion) && Boolean(repoLatestVersion) && installedVersion !== repoLatestVersion;

      // If we have a saved-object based status, return it (augmented with update info).
      // (Backwards compatibility: if not found, fall back to index existence checks below.)
      if (status.status !== 'uninstalled' || status.version || status.failureReason) {
        return {
          ...status,
          latestVersion: repoLatestVersion,
          isUpdateAvailable
        };
      }
      const indexName = (0, _productDocCommon.getSecurityLabsIndexName)(effectiveInferenceId);
      const exists = await this.esClient.indices.exists({
        index: indexName
      });
      if (!exists) return {
        status: 'uninstalled'
      };
      const countResponse = await this.esClient.count({
        index: indexName
      });
      if (countResponse.count === 0) return {
        status: 'uninstalled'
      };

      // Unknown version (legacy install), but installed.
      return {
        status: 'installed',
        latestVersion: repoLatestVersion
      };
    } catch (error) {
      this.log.error(`Error checking Security Labs status: ${error.message}`);
      return {
        status: 'error',
        failureReason: error.message
      };
    }
  }

  /**
   * Ensure Security Labs content is up to date, if currently installed.
   */
  async ensureSecurityLabsUpToDate(params) {
    const {
      inferenceId,
      forceUpdate
    } = params;
    const status = await this.productDocClient.getSecurityLabsInstallationStatus({
      inferenceId
    });
    if (status.status !== 'installed') {
      return;
    }
    const availableVersions = await (0, _steps.fetchSecurityLabsVersions)({
      artifactRepositoryUrl: this.artifactRepositoryUrl
    });
    if (availableVersions.length === 0) {
      return;
    }
    const latest = availableVersions.sort().reverse()[0];
    const installedVersion = status.version;
    if (!forceUpdate && installedVersion && installedVersion === latest) {
      return;
    }
    await this.installSecurityLabs({
      version: latest,
      inferenceId
    });
  }
}
exports.PackageInstaller = PackageInstaller;
const selectVersion = (currentVersion, availableVersions) => {
  return availableVersions.includes(currentVersion) ? currentVersion : (0, _semver.latestVersion)(availableVersions, currentVersion);
};