"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInstallationRoutes = void 0;
var _coreSecurityServer = require("@kbn/core-security-server");
var _configSchema = require("@kbn/config-schema");
var _inferenceCommon = require("@kbn/inference-common");
var _productDocCommon = require("@kbn/product-doc-common");
var _installation = require("../../common/http_api/installation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Schema for resourceType parameter validation.
 */
const resourceTypeSchema = _configSchema.schema.oneOf([_configSchema.schema.literal(_productDocCommon.ResourceTypes.productDoc), _configSchema.schema.literal(_productDocCommon.ResourceTypes.securityLabs)], {
  defaultValue: _productDocCommon.ResourceTypes.productDoc
});
const registerInstallationRoutes = ({
  router,
  getServices
}) => {
  router.get({
    path: _installation.INSTALLATION_STATUS_API_PATH,
    validate: {
      query: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        }),
        resourceType: resourceTypeSchema
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$query, _req$query2;
    const {
      installClient,
      documentationManager
    } = getServices();
    const inferenceId = (_req$query = req.query) === null || _req$query === void 0 ? void 0 : _req$query.inferenceId;
    const resourceType = (_req$query2 = req.query) === null || _req$query2 === void 0 ? void 0 : _req$query2.resourceType;

    // Handle Security Labs status separately
    if (resourceType === _productDocCommon.ResourceTypes.securityLabs) {
      const securityLabsStatus = await documentationManager.getSecurityLabsStatus({
        inferenceId
      });
      return res.ok({
        body: {
          inferenceId,
          resourceType: _productDocCommon.ResourceTypes.securityLabs,
          status: securityLabsStatus.status,
          version: securityLabsStatus.version,
          latestVersion: securityLabsStatus.latestVersion,
          isUpdateAvailable: securityLabsStatus.isUpdateAvailable,
          failureReason: securityLabsStatus.failureReason
        }
      });
    }

    // Default: product documentation status
    const installStatus = await installClient.getInstallationStatus({
      inferenceId
    });
    const {
      status: overallStatus
    } = await documentationManager.getStatus({
      inferenceId
    });
    return res.ok({
      body: {
        inferenceId,
        perProducts: installStatus,
        overall: overallStatus,
        resourceType: _productDocCommon.ResourceTypes.productDoc
      }
    });
  });
  router.post({
    path: _installation.INSTALL_ALL_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        }),
        resourceType: resourceTypeSchema
      })
    },
    options: {
      access: 'internal',
      timeout: {
        idleSocket: 20 * 60 * 1000
      } // install can take time.
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$body, _req$body2;
    const {
      documentationManager
    } = getServices();
    const inferenceId = (_req$body = req.body) === null || _req$body === void 0 ? void 0 : _req$body.inferenceId;
    const resourceType = (_req$body2 = req.body) === null || _req$body2 === void 0 ? void 0 : _req$body2.resourceType;

    // Handle Security Labs installation
    if (resourceType === _productDocCommon.ResourceTypes.securityLabs) {
      await documentationManager.installSecurityLabs({
        request: req,
        wait: true,
        inferenceId
      });
      const securityLabsStatus = await documentationManager.getSecurityLabsStatus({
        inferenceId
      });
      return res.ok({
        body: {
          installed: securityLabsStatus.status === 'installed',
          ...(securityLabsStatus.failureReason ? {
            failureReason: securityLabsStatus.failureReason
          } : {})
        }
      });
    }

    // Default: product documentation installation
    await documentationManager.install({
      request: req,
      force: false,
      wait: true,
      inferenceId
    });

    // check status after installation in case of failure
    const {
      status,
      installStatus
    } = await documentationManager.getStatus({
      inferenceId
    });
    let failureReason = null;
    if (status === 'error' && installStatus) {
      failureReason = Object.values(installStatus).filter(product => product.status === 'error' && product.failureReason).map(product => product.failureReason).join('\n');
    }
    return res.ok({
      body: {
        installed: status === 'installed',
        ...(failureReason ? {
          failureReason
        } : {})
      }
    });
  });
  router.post({
    path: _installation.UPDATE_ALL_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        forceUpdate: _configSchema.schema.boolean({
          defaultValue: false
        }),
        inferenceIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      })
    },
    options: {
      access: 'internal',
      timeout: {
        idleSocket: 20 * 60 * 1000
      } // install can take time.
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$body3, _req$body$inferenceId;
    const {
      forceUpdate
    } = (_req$body3 = req.body) !== null && _req$body3 !== void 0 ? _req$body3 : {};
    const {
      documentationManager
    } = getServices();
    const updated = await documentationManager.updateAll({
      request: req,
      forceUpdate,
      // If inferenceIds is provided, use it, otherwise use all previously installed inference IDs
      inferenceIds: (_req$body$inferenceId = req.body.inferenceIds) !== null && _req$body$inferenceId !== void 0 ? _req$body$inferenceId : []
    });

    // check status after installation in case of failure
    const statuses = await documentationManager.getStatuses({
      inferenceIds: updated.inferenceIds
    });
    return res.ok({
      body: statuses
    });
  });
  router.post({
    path: _installation.UNINSTALL_ALL_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        }),
        resourceType: resourceTypeSchema
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$body4, _req$body6;
    const {
      documentationManager
    } = getServices();
    const resourceType = (_req$body4 = req.body) === null || _req$body4 === void 0 ? void 0 : _req$body4.resourceType;

    // Handle Security Labs uninstallation
    if (resourceType === _productDocCommon.ResourceTypes.securityLabs) {
      var _req$body5;
      await documentationManager.uninstallSecurityLabs({
        request: req,
        wait: true,
        inferenceId: (_req$body5 = req.body) === null || _req$body5 === void 0 ? void 0 : _req$body5.inferenceId
      });
      return res.ok({
        body: {
          success: true
        }
      });
    }

    // Default: product documentation uninstallation
    await documentationManager.uninstall({
      request: req,
      wait: true,
      inferenceId: (_req$body6 = req.body) === null || _req$body6 === void 0 ? void 0 : _req$body6.inferenceId
    });
    return res.ok({
      body: {
        success: true
      }
    });
  });
};
exports.registerInstallationRoutes = registerInstallationRoutes;