"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeAgentIdForTelemetry = normalizeAgentIdForTelemetry;
exports.normalizeToolIdForTelemetry = normalizeToolIdForTelemetry;
var _onechatCommon = require("@kbn/onechat-common");
var _allow_lists = require("@kbn/onechat-server/allow_lists");
var _crypto = require("crypto");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BUILTIN_AGENT_IDS = new Set([_onechatCommon.oneChatDefaultAgentId, ..._allow_lists.AGENT_BUILDER_BUILTIN_AGENTS]);
const BUILTIN_TOOL_IDS = new Set(_allow_lists.AGENT_BUILDER_BUILTIN_TOOLS);
const CUSTOM = 'custom';
const CUSTOM_HASH_PREFIX = `${CUSTOM}-`;
const CUSTOM_HASH_HEX_LENGTH = 16;
function sha256Hex(value) {
  return (0, _crypto.createHash)('sha256').update(value).digest('hex');
}
function toCustomHashedId(value) {
  return `${CUSTOM_HASH_PREFIX}${sha256Hex(value).slice(0, CUSTOM_HASH_HEX_LENGTH)}`;
}

/**
 * Normalizes agent IDs for telemetry to protect user privacy.
 * Built-in agents are reported with their actual ID, custom agents are reported as a stable hashed
 * label (CUSTOM-<sha256_prefix>).
 */
function normalizeAgentIdForTelemetry(agentId) {
  if (!agentId) {
    return undefined;
  }
  return BUILTIN_AGENT_IDS.has(agentId) ? agentId : toCustomHashedId(agentId);
}

/**
 * Normalizes tool IDs for telemetry to protect user privacy.
 * Built-in tools (from AGENT_BUILDER_BUILTIN_TOOLS) are reported with their actual ID,
 * custom/user-created tools are reported as a stable hashed label (CUSTOM-<sha256_prefix>).
 */
function normalizeToolIdForTelemetry(toolId) {
  return BUILTIN_TOOL_IDS.has(toolId) ? toolId : toCustomHashedId(toolId);
}