"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateConfig = validateConfig;
exports.validateConnector = validateConnector;
exports.validateToolName = validateToolName;
var _onechatCommon = require("@kbn/onechat-common");
var _constants = require("@kbn/connector-schemas/mcp/constants");
var _tool_type = require("./tool_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Validates that the connector exists and is an MCP connector.
 */
async function validateConnector({
  actions,
  request,
  connectorId
}) {
  const actionsClient = await actions.getActionsClientWithRequest(request);
  let connector;
  try {
    connector = await actionsClient.get({
      id: connectorId
    });
  } catch (error) {
    throw (0, _onechatCommon.createBadRequestError)(`Connector '${connectorId}' not found or not accessible`);
  }
  if (connector.actionTypeId !== _constants.CONNECTOR_ID) {
    throw (0, _onechatCommon.createBadRequestError)(`Connector '${connectorId}' is not an MCP connector. Expected type '${_constants.CONNECTOR_ID}', got '${connector.actionTypeId}'`);
  }
}

/**
 * Validates that the tool name exists on the MCP server by calling listTools.
 */
async function validateToolName({
  actions,
  request,
  connectorId,
  toolName
}) {
  let tools;
  try {
    const result = await (0, _tool_type.listMcpTools)({
      actions,
      request,
      connectorId
    });
    tools = result.tools;
  } catch (error) {
    // If listTools fails, the connector may not be accessible or the MCP server may be down
    throw (0, _onechatCommon.createBadRequestError)(`Unable to verify tool '${toolName}' on connector '${connectorId}'. ` + `Ensure the connector has successfully connected to the MCP server.`);
  }
  const tool = tools.find(t => t.name === toolName);
  if (!tool) {
    const availableTools = tools.map(t => t.name).join(', ');
    throw (0, _onechatCommon.createBadRequestError)(`Tool '${toolName}' not found on MCP connector '${connectorId}'. Available tools: ${availableTools || 'none'}`);
  }
}

/**
 * Validates MCP tool configuration.
 * Validates:
 * - Connector exists
 * - Connector is of type MCP (.mcp)
 * - Tool name exists on the MCP server
 */
async function validateConfig({
  actions,
  request,
  config
}) {
  await validateConnector({
    actions,
    request,
    connectorId: config.connector_id
  });
  await validateToolName({
    actions,
    request,
    connectorId: config.connector_id,
    toolName: config.tool_name
  });
}