"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _onechatCommon = require("@kbn/onechat-common");
var _spaces = require("../../../../utils/spaces");
var _storage = require("./storage");
var _converters = require("./converters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Client for managing tool health state.
 *
 * Note: This client is primarily used in fire-and-forget mode from the tool registry.
 * Errors are caught and wrapped with context for debugging, but callers using
 * fire-and-forget patterns should catch and ignore errors to avoid blocking tool execution.
 */

const createClient = ({
  space,
  logger,
  esClient
}) => {
  const storage = (0, _storage.createStorage)({
    logger,
    esClient
  });
  return new ToolHealthClientImpl({
    space,
    storage,
    logger
  });
};
exports.createClient = createClient;
class ToolHealthClientImpl {
  constructor({
    space,
    storage,
    logger
  }) {
    (0, _defineProperty2.default)(this, "space", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.space = space;
    this.storage = storage;
    this.logger = logger;
  }
  async get(toolId) {
    try {
      const response = await this.storage.getClient().search({
        track_total_hits: false,
        size: 1,
        terminate_after: 1,
        query: {
          bool: {
            filter: [(0, _spaces.createSpaceDslFilter)(this.space), {
              term: {
                tool_id: toolId
              }
            }]
          }
        }
      });
      const hit = response.hits.hits[0];
      if (!(hit !== null && hit !== void 0 && hit._source)) return undefined;
      return (0, _converters.fromEs)(hit);
    } catch (error) {
      this.logger.error(`Failed to get health state for tool ${toolId}: ${error}`);
      throw (0, _onechatCommon.createInternalError)(`Failed to get health state for tool ${toolId}`, {
        toolId,
        space: this.space,
        cause: error
      });
    }
  }
  async upsert(toolId, params) {
    try {
      var _params$lastCheck, _params$errorMessage, _ref, _params$consecutiveFa;
      const existing = await this.get(toolId);
      const now = new Date().toISOString();
      const document = {
        tool_id: toolId,
        space: this.space,
        status: params.status,
        last_check: (_params$lastCheck = params.lastCheck) !== null && _params$lastCheck !== void 0 ? _params$lastCheck : now,
        error_message: (_params$errorMessage = params.errorMessage) !== null && _params$errorMessage !== void 0 ? _params$errorMessage : '',
        consecutive_failures: (_ref = (_params$consecutiveFa = params.consecutiveFailures) !== null && _params$consecutiveFa !== void 0 ? _params$consecutiveFa : existing === null || existing === void 0 ? void 0 : existing.consecutiveFailures) !== null && _ref !== void 0 ? _ref : 0,
        updated_at: now
      };

      // Use tool_id + space as a deterministic document ID
      const docId = `${this.space}:${toolId}`;
      await this.storage.getClient().index({
        id: docId,
        document
      });
      this.logger.debug(`Updated health state for tool ${toolId}: ${params.status}`);
      return {
        toolId,
        status: params.status,
        lastCheck: document.last_check,
        errorMessage: params.errorMessage,
        consecutiveFailures: document.consecutive_failures
      };
    } catch (error) {
      // Don't re-wrap if already an internal error (from get() call)
      if (error && typeof error === 'object' && 'code' in error) {
        throw error;
      }
      this.logger.error(`Failed to upsert health state for tool ${toolId}: ${error}`);
      throw (0, _onechatCommon.createInternalError)(`Failed to upsert health state for tool ${toolId}`, {
        toolId,
        space: this.space,
        status: params.status,
        cause: error
      });
    }
  }
  async delete(toolId) {
    try {
      const docId = `${this.space}:${toolId}`;
      const result = await this.storage.getClient().delete({
        id: docId
      });
      return result.result === 'deleted';
    } catch (error) {
      // Fire-and-forget friendly: log and return false instead of throwing
      this.logger.debug(`Failed to delete health state for tool ${toolId}: ${error}`);
      return false;
    }
  }
  async listBySpace() {
    try {
      const response = await this.storage.getClient().search({
        size: 1000,
        track_total_hits: false,
        query: {
          bool: {
            filter: [(0, _spaces.createSpaceDslFilter)(this.space)]
          }
        }
      });
      return response.hits.hits.filter(hit => !!hit._source).map(hit => (0, _converters.fromEs)(hit));
    } catch (error) {
      this.logger.error(`Failed to list health states for space ${this.space}: ${error}`);
      throw (0, _onechatCommon.createInternalError)(`Failed to list health states for space ${this.space}`, {
        space: this.space,
        cause: error
      });
    }
  }
  async recordSuccess(toolId) {
    return this.upsert(toolId, {
      status: 'healthy',
      lastCheck: new Date().toISOString(),
      errorMessage: undefined,
      consecutiveFailures: 0
    });
  }
  async recordFailure(toolId, errorMessage) {
    var _existing$consecutive;
    const existing = await this.get(toolId);
    const consecutiveFailures = ((_existing$consecutive = existing === null || existing === void 0 ? void 0 : existing.consecutiveFailures) !== null && _existing$consecutive !== void 0 ? _existing$consecutive : 0) + 1;
    return this.upsert(toolId, {
      status: 'failed',
      lastCheck: new Date().toISOString(),
      errorMessage,
      consecutiveFailures
    });
  }
}