"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.roundToLangchain = exports.createToolCallMessages = exports.conversationToLangchainMessages = void 0;
var _messages = require("@langchain/core/messages");
var _onechatCommon = require("@kbn/onechat-common");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _utils = require("@kbn/onechat-genai-utils/tools/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a conversation to langchain format
 */
const conversationToLangchainMessages = ({
  conversation,
  ignoreSteps = false
}) => {
  const messages = [];
  for (const round of conversation.previousRounds) {
    messages.push(...roundToLangchain(round, {
      ignoreSteps
    }));
  }
  messages.push(formatRoundInput({
    input: conversation.nextInput
  }));
  return messages;
};
exports.conversationToLangchainMessages = conversationToLangchainMessages;
const roundToLangchain = (round, {
  ignoreSteps = false
} = {}) => {
  const messages = [];

  // user message
  messages.push(formatRoundInput({
    input: round.input
  }));

  // steps
  if (!ignoreSteps) {
    for (const step of round.steps) {
      if ((0, _onechatCommon.isToolCallStep)(step)) {
        messages.push(...createToolCallMessages(step));
      }
    }
  }

  // assistant response
  messages.push(formatAssistantResponse({
    response: round.response
  }));
  return messages;
};
exports.roundToLangchain = roundToLangchain;
const formatRoundInput = ({
  input
}) => {
  const {
    message,
    attachments
  } = input;
  let content = message;
  if (attachments.length > 0) {
    content += `\n\n
<attachments>
${attachments.map(attachment => formatAttachment({
      attachment,
      indent: 2
    })).join('\n')}
</attachments>
`;
  }
  return (0, _langchain.createUserMessage)(content);
};
const formatAttachment = ({
  attachment,
  indent = 0
}) => {
  return (0, _utils.generateXmlTree)({
    tagName: 'attachment',
    attributes: {
      type: attachment.attachment.type,
      id: attachment.attachment.id
    },
    children: [attachment.representation.value]
  }, {
    initialIndentLevel: indent,
    escapeContent: false
  });
};
const formatAssistantResponse = ({
  response
}) => {
  return (0, _langchain.createAIMessage)(response.message);
};
const createToolCallMessages = toolCall => {
  const toolName = (0, _langchain.sanitizeToolId)(toolCall.tool_id);
  const toolCallMessage = new _messages.AIMessage({
    content: '',
    tool_calls: [{
      id: toolCall.tool_call_id,
      name: toolName,
      args: toolCall.params,
      type: 'tool_call'
    }]
  });
  const toolResultMessage = new _messages.ToolMessage({
    tool_call_id: toolCall.tool_call_id,
    content: JSON.stringify({
      results: toolCall.results
    })
  });
  return [toolCallMessage, toolResultMessage];
};
exports.createToolCallMessages = createToolCallMessages;