"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _onechatCommon = require("@kbn/onechat-common");
var _agents = require("@kbn/onechat-common/agents");
var _agent_source = require("./agent_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// internal definition for our agents

const createAgentRegistry = opts => {
  return new AgentRegistryImpl(opts);
};
exports.createAgentRegistry = createAgentRegistry;
class AgentRegistryImpl {
  constructor({
    request,
    spaceId,
    persistedProvider,
    builtinProvider,
    uiSettings,
    savedObjects
  }) {
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "persistedProvider", void 0);
    (0, _defineProperty2.default)(this, "builtinProvider", void 0);
    (0, _defineProperty2.default)(this, "uiSettings", void 0);
    (0, _defineProperty2.default)(this, "savedObjects", void 0);
    this.request = request;
    this.spaceId = spaceId;
    this.persistedProvider = persistedProvider;
    this.builtinProvider = builtinProvider;
    this.uiSettings = uiSettings;
    this.savedObjects = savedObjects;
  }
  get orderedProviders() {
    return [this.builtinProvider, this.persistedProvider];
  }
  async has(agentId) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(agentId)) {
        return true;
      }
    }
    return false;
  }
  async get(agentId) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(agentId)) {
        const agent = await provider.get(agentId);
        if (!(await this.isAvailable(agent))) {
          throw (0, _onechatCommon.createBadRequestError)(`Agent ${agentId} is not available`);
        }
        return agent;
      }
    }
    throw (0, _onechatCommon.createAgentNotFoundError)({
      agentId
    });
  }
  async list(opts) {
    const allAgents = [];
    for (const provider of this.orderedProviders) {
      const providerAgents = await provider.list(opts);
      for (const agent of providerAgents) {
        if (await this.isAvailable(agent)) {
          allAgents.push(agent);
        }
      }
    }
    return allAgents;
  }
  async create(createRequest) {
    const {
      id: agentId
    } = createRequest;
    const validationError = (0, _agents.validateAgentId)({
      agentId,
      builtIn: false
    });
    if (validationError) {
      throw (0, _onechatCommon.createBadRequestError)(`Invalid agent id: "${agentId}": ${validationError}`);
    }
    if (await this.has(agentId)) {
      throw (0, _onechatCommon.createBadRequestError)(`Agent with id ${agentId} already exists`);
    }
    return this.persistedProvider.create(createRequest);
  }
  async update(agentId, update) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(agentId)) {
        if ((0, _agent_source.isReadonlyProvider)(provider)) {
          throw (0, _onechatCommon.createBadRequestError)(`Agent ${agentId} is read-only and can't be updated`);
        } else {
          return provider.update(agentId, update);
        }
      }
    }
    throw (0, _onechatCommon.createAgentNotFoundError)({
      agentId
    });
  }
  async delete({
    id: agentId
  }) {
    for (const provider of this.orderedProviders) {
      if (await provider.has(agentId)) {
        if ((0, _agent_source.isReadonlyProvider)(provider)) {
          throw (0, _onechatCommon.createBadRequestError)(`Agent ${agentId} is read-only and can't be deleted`);
        } else {
          return provider.delete(agentId);
        }
      }
    }
    throw (0, _onechatCommon.createAgentNotFoundError)({
      agentId
    });
  }
  async isAvailable(agent) {
    const soClient = this.savedObjects.getScopedClient(this.request);
    const uiSettingsClient = this.uiSettings.asScopedToClient(soClient);
    const context = {
      spaceId: this.spaceId,
      request: this.request,
      uiSettings: uiSettingsClient
    };
    const status = await agent.isAvailable(context);
    return status.status === 'available';
  }
}