"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInternalToolsRoutes = registerInternalToolsRoutes;
var _configSchema = require("@kbn/config-schema");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _onechatCommon = require("@kbn/onechat-common");
var _constants = require("@kbn/connector-schemas/mcp/constants");
var _tools = require("@kbn/onechat-common/tools");
var _wrap_handler = require("../wrap_handler");
var _validate_configuration = require("../../services/tools/tool_types/mcp/validate_configuration");
var _features = require("../../../common/features");
var _constants2 = require("../../../common/constants");
var _utils = require("../../services/tools/utils");
var _utils2 = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerInternalToolsRoutes({
  router,
  coreSetup,
  getInternalServices,
  logger,
  pluginsSetup: {
    workflowsManagement
  }
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // bulk delete tools
  router.post({
    path: `${_constants2.internalApiPath}/tools/_bulk_delete`,
    validate: {
      body: _configSchema.schema.object({
        ids: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      ids
    } = request.body;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const deleteResults = await Promise.allSettled(ids.map(id => registry.delete(id)));
    const results = deleteResults.map((result, index) => {
      if (result.status !== 'fulfilled') {
        var _result$reason$toJSON, _result$reason$toJSON2, _result$reason;
        return {
          toolId: ids[index],
          success: false,
          reason: (_result$reason$toJSON = (_result$reason$toJSON2 = (_result$reason = result.reason).toJSON) === null || _result$reason$toJSON2 === void 0 ? void 0 : _result$reason$toJSON2.call(_result$reason)) !== null && _result$reason$toJSON !== void 0 ? _result$reason$toJSON : {
            error: {
              message: 'Unknown error'
            }
          }
        };
      }
      return {
        toolId: ids[index],
        success: true
      };
    });
    return response.ok({
      body: {
        results
      }
    });
  }));

  // bulk create MCP tools from connector (internal)
  router.post({
    path: `${_constants2.internalApiPath}/tools/_bulk_create_mcp`,
    validate: {
      body: _configSchema.schema.object({
        connector_id: _configSchema.schema.string(),
        tools: _configSchema.schema.arrayOf(_configSchema.schema.object({
          name: _configSchema.schema.string(),
          description: _configSchema.schema.maybe(_configSchema.schema.string())
        })),
        namespace: _configSchema.schema.maybe(_configSchema.schema.string()),
        tags: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
          defaultValue: []
        }),
        skip_existing: _configSchema.schema.boolean({
          defaultValue: true
        })
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      connector_id: connectorId,
      tools,
      namespace,
      tags,
      skip_existing: skipExisting
    } = request.body;
    const {
      tools: toolService
    } = getInternalServices();
    const [, {
      actions
    }] = await coreSetup.getStartServices();
    const registry = await toolService.getRegistry({
      request
    });

    // Validate namespace if provided (must be valid tool ID segment)
    if (namespace) {
      const namespaceError = (0, _onechatCommon.validateToolId)({
        toolId: namespace,
        builtIn: false
      });
      if (namespaceError) {
        return response.badRequest({
          body: {
            message: `Invalid namespace: ${namespaceError}`
          }
        });
      }
    }

    // Validate connector is MCP type
    await (0, _validate_configuration.validateConnector)({
      actions,
      request,
      connectorId
    });

    // Precompute tool metadata (toolId, mcpToolName) once per tool
    // MCP tool names are server-generated and typically well-formed (e.g., snake_case)
    // We just lowercase them; validation in registry.create() handles edge cases
    const toolsWithIds = tools.map(tool => {
      const toolName = tool.name.toLowerCase();
      const toolId = namespace ? `${namespace}.${toolName}` : toolName;
      return {
        toolId,
        mcpToolName: tool.name,
        description: tool.description
      };
    });

    // Process tools in parallel using Promise.allSettled (matches bulk delete pattern)
    const createResults = await Promise.allSettled(toolsWithIds.map(async ({
      toolId,
      mcpToolName,
      description
    }) => {
      // Check if tool already exists
      const exists = await registry.has(toolId);
      if (exists && skipExisting) {
        return {
          toolId,
          mcpToolName,
          skipped: true
        };
      }

      // Create the MCP tool
      await registry.create({
        id: toolId,
        type: _onechatCommon.ToolType.mcp,
        description: description !== null && description !== void 0 ? description : '',
        tags,
        configuration: {
          connector_id: connectorId,
          tool_name: mcpToolName
        }
      });
      return {
        toolId,
        mcpToolName,
        skipped: false
      };
    }));

    // Map results to response format (matches bulk delete pattern)
    const results = createResults.map((result, index) => {
      const {
        toolId,
        mcpToolName
      } = toolsWithIds[index];
      if (result.status === 'rejected') {
        var _result$reason$toJSON3, _result$reason2, _result$reason2$toJSO, _result$reason$messag, _result$reason3;
        return {
          toolId,
          mcpToolName,
          success: false,
          reason: (_result$reason$toJSON3 = (_result$reason2 = result.reason) === null || _result$reason2 === void 0 ? void 0 : (_result$reason2$toJSO = _result$reason2.toJSON) === null || _result$reason2$toJSO === void 0 ? void 0 : _result$reason2$toJSO.call(_result$reason2)) !== null && _result$reason$toJSON3 !== void 0 ? _result$reason$toJSON3 : {
            error: {
              message: (_result$reason$messag = (_result$reason3 = result.reason) === null || _result$reason3 === void 0 ? void 0 : _result$reason3.message) !== null && _result$reason$messag !== void 0 ? _result$reason$messag : 'Unknown error'
            }
          }
        };
      }
      if (result.value.skipped) {
        return {
          toolId: result.value.toolId,
          mcpToolName: result.value.mcpToolName,
          success: true,
          skipped: true
        };
      }
      return {
        toolId: result.value.toolId,
        mcpToolName: result.value.mcpToolName,
        success: true
      };
    });

    // Compute summary counts
    const summary = results.reduce((acc, r) => {
      if (!r.success) acc.failed++;else if ('skipped' in r && r.skipped) acc.skipped++;else acc.created++;
      return acc;
    }, {
      total: results.length,
      created: 0,
      skipped: 0,
      failed: 0
    });
    return response.ok({
      body: {
        results,
        summary
      }
    });
  }));

  // validate namespace for MCP tool import (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_validate_namespace`,
    validate: {
      query: _configSchema.schema.object({
        namespace: _configSchema.schema.string(),
        connector_id: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      namespace,
      connector_id: connectorId
    } = request.query;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const allTools = await registry.list({});
    const toolsInNamespace = allTools.filter(tool => {
      const lastDotIndex = tool.id.lastIndexOf('.');
      if (lastDotIndex > 0) {
        const toolNamespace = tool.id.substring(0, lastDotIndex);
        return toolNamespace === namespace;
      }
      return false;
    });
    if (toolsInNamespace.length === 0) {
      return response.ok({
        body: {
          isValid: true,
          conflictingNamespaces: []
        }
      });
    }

    // If connectorId is provided, check if all tools in the namespace belong to the same connector
    // This allows reusing a namespace for the same MCP server
    if (connectorId) {
      const allToolsBelongToSameConnector = toolsInNamespace.every(tool => {
        if ((0, _tools.isMcpTool)(tool)) {
          return tool.configuration.connector_id === connectorId;
        }
        return false;
      });
      if (allToolsBelongToSameConnector) {
        return response.ok({
          body: {
            isValid: true,
            conflictingNamespaces: []
          }
        });
      }
    }
    return response.ok({
      body: {
        isValid: false,
        conflictingNamespaces: [namespace]
      }
    });
  }));

  // resolve search sources (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_resolve_search_sources`,
    validate: {
      query: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const esClient = (await ctx.core).elasticsearch.client.asCurrentUser;
    const {
      pattern
    } = request.query;
    const {
      indices,
      aliases,
      data_streams: dataStreams
    } = await (0, _onechatGenaiUtils.listSearchSources)({
      pattern,
      includeHidden: false,
      includeKibanaIndices: false,
      excludeIndicesRepresentedAsAlias: true,
      excludeIndicesRepresentedAsDatastream: true,
      esClient
    });
    const results = [...indices.map(i => ({
      type: 'index',
      name: i.name
    })), ...aliases.map(a => ({
      type: 'alias',
      name: a.name
    })), ...dataStreams.map(d => ({
      type: 'data_stream',
      name: d.name
    }))];
    return response.ok({
      body: {
        results,
        total: results.length
      }
    });
  }));

  // list available tool types (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_types_info`,
    validate: false,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools
    } = getInternalServices();
    const toolTypes = tools.getToolDefinitions();
    return response.ok({
      body: {
        toolTypes: (0, _utils.getToolTypeInfo)(toolTypes)
      }
    });
  }));

  // list workflows (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_list_workflows`,
    validate: {
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 1
        }),
        limit: _configSchema.schema.number({
          defaultValue: 10000
        })
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    if (!workflowsManagement) {
      return response.ok({
        body: {
          results: []
        }
      });
    }
    const currentSpace = (await ctx.onechat).spaces.getSpaceId();
    const {
      results
    } = await workflowsManagement.management.getWorkflows({
      page: request.query.page,
      size: request.query.limit,
      enabled: [true]
    }, currentSpace);
    return response.ok({
      body: {
        results: results.map(workflow => ({
          id: workflow.id,
          name: workflow.name,
          description: workflow.description
        }))
      }
    });
  }));

  // get workflow (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_get_workflow/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    if (!workflowsManagement) {
      return response.ok({
        body: {
          id: '',
          name: '',
          description: ''
        }
      });
    }
    const currentSpace = (await ctx.onechat).spaces.getSpaceId();
    const workflow = await workflowsManagement.management.getWorkflow(request.params.id, currentSpace);
    return response.ok({
      body: {
        id: workflow.id,
        name: workflow.name,
        description: workflow.description
      }
    });
  }));

  // list all tool health statuses for the current space (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_health`,
    validate: false,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools
    } = getInternalServices();
    const healthClient = tools.getHealthClient({
      request
    });
    const healthStates = await healthClient.listBySpace();
    return response.ok({
      body: {
        results: healthStates
      }
    });
  }));

  // get health status for a specific tool (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/{toolId}/_health`,
    validate: {
      params: _configSchema.schema.object({
        toolId: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools
    } = getInternalServices();
    const healthClient = tools.getHealthClient({
      request
    });
    const health = await healthClient.get(request.params.toolId);
    if (!health) {
      return response.notFound({
        body: {
          message: `No health data found for tool '${request.params.toolId}'`
        }
      });
    }
    return response.ok({
      body: {
        health
      }
    });
  }));

  // list connectors (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_list_connectors`,
    validate: {
      query: _configSchema.schema.object({
        type: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const [, pluginsStart] = await coreSetup.getStartServices();
    const actionsClient = await pluginsStart.actions.getActionsClientWithRequest(request);
    const allConnectors = await actionsClient.getAll();
    const {
      type
    } = request.query;
    const connectors = allConnectors.filter(connector => type ? connector.actionTypeId === type : true).map(_utils2.toConnectorItem);
    return response.ok({
      body: {
        connectors
      }
    });
  }));

  // get connector by ID (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_get_connector/{connectorId}`,
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const [, pluginsStart] = await coreSetup.getStartServices();
    const actionsClient = await pluginsStart.actions.getActionsClientWithRequest(request);
    const {
      connectorId
    } = request.params;
    const connector = await actionsClient.get({
      id: connectorId
    });
    return response.ok({
      body: {
        connector: (0, _utils2.toConnectorItem)(connector)
      }
    });
  }));

  // list MCP tools (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_list_mcp_tools`,
    validate: {
      query: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    var _executeResult$data$t, _executeResult$data;
    const [, pluginsStart] = await coreSetup.getStartServices();
    const actionsClient = await pluginsStart.actions.getActionsClientWithRequest(request);
    const {
      connectorId
    } = request.query;
    const connector = await actionsClient.get({
      id: connectorId
    });
    if (connector.actionTypeId !== _constants.CONNECTOR_ID) {
      response.badRequest({
        body: {
          message: `Connector '${connectorId}' is not an MCP connector. Expected type '${_constants.CONNECTOR_ID}', got '${connector.actionTypeId}'`
        }
      });
    }
    const executeResult = await actionsClient.execute({
      actionId: request.query.connectorId,
      params: {
        subAction: 'listTools'
      }
    });
    if (executeResult.status === 'error') {
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to list MCP tools for connector '${connectorId}': ${executeResult.message}`
        }
      });
    }
    const mcpTools = (_executeResult$data$t = (_executeResult$data = executeResult.data) === null || _executeResult$data === void 0 ? void 0 : _executeResult$data.tools) !== null && _executeResult$data$t !== void 0 ? _executeResult$data$t : [];
    return response.ok({
      body: {
        mcpTools
      }
    });
  }));

  // list health status for all MCP tools (internal)
  router.get({
    path: `${_constants2.internalApiPath}/tools/_mcp_health`,
    validate: false,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    }
  }, wrapHandler(async (ctx, request, response) => {
    const [, pluginsStart] = await coreSetup.getStartServices();
    const actionsClient = await pluginsStart.actions.getActionsClientWithRequest(request);
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const healthClient = toolService.getHealthClient({
      request
    });
    const allTools = await registry.list({});
    const mcpTools = allTools.filter(tool => (0, _tools.isMcpTool)(tool));
    if (mcpTools.length === 0) {
      return response.ok({
        body: {
          results: []
        }
      });
    }
    const healthStates = await healthClient.listBySpace();
    const healthByToolId = new Map(healthStates.map(healthState => [healthState.toolId, healthState]));
    const connectorIds = [...new Set(mcpTools.map(tool => tool.configuration.connector_id))];
    const connectorResults = await Promise.allSettled(connectorIds.map(async connectorId => {
      var _executeResult$data$t2, _executeResult$data2;
      const connector = await actionsClient.get({
        id: connectorId
      });
      const executeResult = await actionsClient.execute({
        actionId: connectorId,
        params: {
          subAction: 'listTools'
        }
      });
      return {
        connectorId,
        connector,
        mcpServerTools: executeResult.status === 'ok' ? (_executeResult$data$t2 = (_executeResult$data2 = executeResult.data) === null || _executeResult$data2 === void 0 ? void 0 : _executeResult$data2.tools) !== null && _executeResult$data$t2 !== void 0 ? _executeResult$data$t2 : [] : [],
        listToolsError: executeResult.status === 'error' ? executeResult.message : undefined
      };
    }));
    const connectorDataMap = new Map();
    for (let i = 0; i < connectorIds.length; i++) {
      const connectorId = connectorIds[i];
      const result = connectorResults[i];
      if (result.status === 'fulfilled') {
        connectorDataMap.set(connectorId, {
          exists: true,
          mcpServerTools: result.value.mcpServerTools.map(mcpServerTool => mcpServerTool.name),
          listToolsError: result.value.listToolsError
        });
      } else {
        connectorDataMap.set(connectorId, {
          exists: false,
          mcpServerTools: []
        });
      }
    }
    const results = mcpTools.map(tool => {
      const toolConnectorId = tool.configuration.connector_id;
      const mcpToolName = tool.configuration.tool_name;
      const connectorData = connectorDataMap.get(toolConnectorId);
      const toolHealth = healthByToolId.get(tool.id);
      let status = 'healthy';
      let errorMessage;

      // Check connector exists
      if (!(connectorData !== null && connectorData !== void 0 && connectorData.exists)) {
        status = 'connector_not_found';
        errorMessage = `Connector '${toolConnectorId}' not found`;
      }
      // Check if listing MCP tools failed
      else if (connectorData.listToolsError) {
        status = 'list_tools_failed';
        errorMessage = connectorData.listToolsError;
      }
      // Check if the specific MCP tool exists on the server
      else if (!connectorData.mcpServerTools.includes(mcpToolName)) {
        status = 'tool_not_found';
        errorMessage = `Tool '${mcpToolName}' not found on MCP server`;
      }
      // Check if the tool has failed health checks
      else if (toolHealth && toolHealth.status !== 'healthy') {
        status = 'tool_unhealthy';
        errorMessage = toolHealth.errorMessage;
      }
      return {
        toolId: tool.id,
        connectorId: toolConnectorId,
        mcpToolName,
        status,
        errorMessage
      };
    });
    return response.ok({
      body: {
        results
      }
    });
  }));
}