"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConversationRoutes = registerConversationRoutes;
var _configSchema = require("@kbn/config-schema");
var _nodePath = _interopRequireDefault(require("node:path"));
var _wrap_handler = require("./wrap_handler");
var _features = require("../../common/features");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerConversationRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // List conversations
  router.versioned.get({
    path: `${_constants.publicApiPath}/conversations`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'List conversations',
    description: 'List all conversations for a user. Use the optional agent ID to filter conversations by a specific agent.',
    options: {
      tags: ['conversation', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        query: _configSchema.schema.object({
          agent_id: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Optional agent ID to filter conversations by a specific agent.'
            }
          }))
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/conversations_list.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      conversations: conversationsService
    } = getInternalServices();
    const {
      agent_id: agentId
    } = request.query;
    const client = await conversationsService.getScopedClient({
      request
    });
    const conversations = await client.list({
      agentId
    });
    return response.ok({
      body: {
        results: conversations
      }
    });
  }));

  // Get conversation by ID
  router.versioned.get({
    path: `${_constants.publicApiPath}/conversations/{conversation_id}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Get conversation by ID',
    description: 'Get a specific conversation by ID. Use this endpoint to retrieve the complete conversation history including all messages and metadata.',
    options: {
      tags: ['conversation', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          conversation_id: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the conversation to retrieve.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/conversations_get_by_id.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      conversations: conversationsService
    } = getInternalServices();
    const {
      conversation_id: conversationId
    } = request.params;
    const client = await conversationsService.getScopedClient({
      request
    });
    const conversation = await client.get(conversationId);
    return response.ok({
      body: conversation
    });
  }));

  // delete conversation by ID
  router.versioned.delete({
    path: `${_constants.publicApiPath}/conversations/{conversation_id}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Delete conversation by ID',
    description: 'Delete a conversation by ID. This action cannot be undone.',
    options: {
      tags: ['conversation', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          conversation_id: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the conversation to delete.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/conversations_delete.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      conversations: conversationsService
    } = getInternalServices();
    const {
      conversation_id: conversationId
    } = request.params;
    const client = await conversationsService.getScopedClient({
      request
    });
    const status = await client.delete(conversationId);
    return response.ok({
      body: {
        success: status
      }
    });
  }));
}