"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConversationActions = void 0;
var _react = require("react");
var _immer = _interopRequireDefault(require("immer"));
var _useLocalStorage = _interopRequireDefault(require("react-use/lib/useLocalStorage"));
var _onechatCommon = require("@kbn/onechat-common");
var _query_keys = require("../../query_keys");
var _storage_keys = require("../../storage_keys");
var _new_conversation = require("../../utils/new_conversation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createConversationActions = ({
  conversationId,
  queryClient,
  setAgentIdStorage,
  conversationsService,
  onConversationCreated,
  onDeleteConversation
}) => {
  const queryKey = _query_keys.queryKeys.conversations.byId(conversationId !== null && conversationId !== void 0 ? conversationId : _new_conversation.newConversationId);
  const setConversation = updater => {
    queryClient.setQueryData(queryKey, updater);
  };
  const setCurrentRound = updater => {
    setConversation((0, _immer.default)(draft => {
      var _draft$rounds;
      const round = draft === null || draft === void 0 ? void 0 : (_draft$rounds = draft.rounds) === null || _draft$rounds === void 0 ? void 0 : _draft$rounds.at(-1);
      if (round) {
        updater(round);
      }
    }));
  };
  return {
    removeNewConversationQuery: () => {
      queryClient.removeQueries({
        queryKey: _query_keys.queryKeys.conversations.byId(_new_conversation.newConversationId)
      });
    },
    invalidateConversation: () => {
      queryClient.invalidateQueries({
        queryKey
      });
    },
    addOptimisticRound: ({
      userMessage,
      attachments
    }) => {
      setConversation((0, _immer.default)(draft => {
        var _attachments$map;
        const optimisticAttachments = (_attachments$map = attachments === null || attachments === void 0 ? void 0 : attachments.map((attachment, idx) => ({
          id: `pending-attachment-${idx}`,
          ...attachment
        }))) !== null && _attachments$map !== void 0 ? _attachments$map : [];
        const nextRound = (0, _new_conversation.createNewRound)({
          userMessage,
          attachments: optimisticAttachments
        });
        if (!draft) {
          const newConversation = (0, _new_conversation.createNewConversation)();
          newConversation.rounds.push(nextRound);
          return newConversation;
        }
        draft.rounds.push(nextRound);
      }));
    },
    removeOptimisticRound: () => {
      setConversation((0, _immer.default)(draft => {
        var _draft$rounds2;
        draft === null || draft === void 0 ? void 0 : (_draft$rounds2 = draft.rounds) === null || _draft$rounds2 === void 0 ? void 0 : _draft$rounds2.pop();
      }));
    },
    setAgentId: agentId => {
      // We allow to change agent only at the start of the conversation
      if (conversationId) {
        return;
      }
      setConversation((0, _immer.default)(draft => {
        if (!draft) {
          const newConversation = (0, _new_conversation.createNewConversation)();
          newConversation.agent_id = agentId;
          return newConversation;
        }
        draft.agent_id = agentId;
      }));
      setAgentIdStorage(agentId);
    },
    addReasoningStep: ({
      step
    }) => {
      setCurrentRound(round => {
        round.steps.push(step);
      });
    },
    addToolCall: ({
      step
    }) => {
      setCurrentRound(round => {
        round.steps.push(step);
      });
    },
    setToolCallProgress: ({
      progress,
      toolCallId
    }) => {
      setCurrentRound(round => {
        const step = round.steps.filter(_onechatCommon.isToolCallStep).find(s => s.tool_call_id === toolCallId);
        if (step) {
          if (!step.progression) {
            step.progression = [];
          }
          step.progression.push(progress);
        }
      });
    },
    setToolCallResult: ({
      results,
      toolCallId
    }) => {
      setCurrentRound(round => {
        const step = round.steps.filter(_onechatCommon.isToolCallStep).find(s => s.tool_call_id === toolCallId);
        if (step) {
          step.results = results;
        }
      });
    },
    setAssistantMessage: ({
      assistantMessage
    }) => {
      setCurrentRound(round => {
        round.response.message = assistantMessage;
      });
    },
    addAssistantMessageChunk: ({
      messageChunk
    }) => {
      setCurrentRound(round => {
        round.response.message += messageChunk;
      });
    },
    setTimeToFirstToken: ({
      timeToFirstToken
    }) => {
      setCurrentRound(round => {
        round.time_to_first_token = timeToFirstToken;
      });
    },
    onConversationCreated: ({
      conversationId: id,
      title
    }) => {
      const current = queryClient.getQueryData(queryKey);
      if (!current) {
        throw new Error('Conversation not created');
      }

      // Update individual conversation cache (with rounds)
      queryClient.setQueryData(_query_keys.queryKeys.conversations.byId(id), (0, _immer.default)(current, draft => {
        draft.id = id;
        draft.title = title;
      }));

      // Invalidate conversation list to get updated data from server
      queryClient.invalidateQueries({
        queryKey: _query_keys.queryKeys.conversations.all
      });

      // Call provider-specific callback if provided
      if (onConversationCreated) {
        onConversationCreated({
          conversationId: id,
          title
        });
      }
    },
    deleteConversation: async id => {
      await conversationsService.delete({
        conversationId: id
      });

      // Check if we're deleting the current conversation
      const isCurrentConversation = conversationId === id;
      queryClient.removeQueries({
        queryKey: _query_keys.queryKeys.conversations.byId(id)
      });
      queryClient.invalidateQueries({
        queryKey: _query_keys.queryKeys.conversations.all
      });

      // Call provider-specific callback if provided
      if (onDeleteConversation) {
        onDeleteConversation({
          id,
          isCurrentConversation
        });
      }
    },
    renameConversation: async (id, title) => {
      await conversationsService.rename({
        conversationId: id,
        title
      });

      // Update the conversation in cache if it exists
      const conversationQueryKey = _query_keys.queryKeys.conversations.byId(id);
      const currentConversation = queryClient.getQueryData(conversationQueryKey);
      if (currentConversation) {
        queryClient.setQueryData(conversationQueryKey, (0, _immer.default)(currentConversation, draft => {
          draft.title = title;
        }));
      }

      // Invalidate conversation list to get updated data from server
      queryClient.invalidateQueries({
        queryKey: _query_keys.queryKeys.conversations.all
      });
    }
  };
};
const useConversationActions = ({
  conversationId,
  queryClient,
  conversationsService,
  onConversationCreated,
  onDeleteConversation
}) => {
  const [, setAgentIdStorage] = (0, _useLocalStorage.default)(_storage_keys.storageKeys.agentId);
  const conversationActions = (0, _react.useMemo)(() => createConversationActions({
    conversationId,
    queryClient,
    setAgentIdStorage,
    conversationsService,
    onConversationCreated,
    onDeleteConversation
  }), [conversationId, queryClient, setAgentIdStorage, conversationsService, onConversationCreated, onDeleteConversation]);
  return conversationActions;
};
exports.useConversationActions = useConversationActions;