"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findToolResult = exports.createTagParser = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createTagParser = config => {
  return () => {
    const extractAttribute = (value, attr) => {
      var _value$match;
      const regex = new RegExp(`${attr}="([^"]*)"`, 'i');
      return (_value$match = value.match(regex)) === null || _value$match === void 0 ? void 0 : _value$match[1];
    };
    const tagRegex = new RegExp(`<${config.tagName}\\b[^>]*\\/?>`, 'gi');
    const visitParent = parent => {
      for (let index = 0; index < parent.children.length; index++) {
        const child = parent.children[index];
        if ('children' in child) {
          visitParent(child);
        }
        if (child.type !== 'html') {
          continue; // terminate iteration if not html node
        }
        const rawValue = child.value;
        if (!rawValue) {
          continue; // terminate iteration if no value attribute
        }
        const trimmedValue = rawValue.trim();
        if (!trimmedValue.toLowerCase().startsWith(`<${config.tagName}`)) {
          continue; // terminate iteration if not starting with tag
        }
        const matches = Array.from(trimmedValue.matchAll(tagRegex));
        if (matches.length === 0) {
          continue; // terminate iteration if no matches found
        }
        const parsedAttributes = matches.map(match => config.getAttributes(match[0], extractAttribute));
        const leftoverContent = trimmedValue.replace(tagRegex, '').trim();
        config.assignAttributes(child, parsedAttributes[0]);
        if (parsedAttributes.length === 1 || leftoverContent.length > 0) {
          continue;
        }
        const additionalNodes = parsedAttributes.slice(1).map(attributes => config.createNode(attributes, child.position));
        const siblings = parent.children;
        siblings.splice(index + 1, 0, ...additionalNodes);
        index += additionalNodes.length;
        continue;
      }
    };
    return tree => {
      if ('children' in tree) {
        visitParent(tree);
      }
    };
  };
};
exports.createTagParser = createTagParser;
const findToolResult = (steps, toolResultId, resultType) => {
  return steps.filter(s => s.type === 'tool_call').flatMap(s => s.type === 'tool_call' && s.results || []).find(r => r.type === resultType && r.tool_result_id === toolResultId);
};
exports.findToolResult = findToolResult;