"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.listSearchSources = void 0;
var _lodash = require("lodash");
var _onechatCommon = require("@kbn/onechat-common");
var _esErrors = require("@kbn/es-errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const kibanaIndicesExclusionPattern = '-.*';

/**
 * List the search sources (indices, aliases and datastreams) matching a given index pattern,
 * using the `_resolve_index` API.
 */
const listSearchSources = async ({
  pattern,
  perTypeLimit = 100,
  includeHidden = false,
  includeKibanaIndices = false,
  excludeIndicesRepresentedAsAlias = true,
  excludeIndicesRepresentedAsDatastream = true,
  esClient
}) => {
  try {
    const resolveRes = await esClient.indices.resolveIndex({
      name: includeKibanaIndices ? [pattern] : [pattern, kibanaIndicesExclusionPattern],
      allow_no_indices: true,
      expand_wildcards: includeHidden ? ['open', 'hidden'] : ['open']
    });

    // data streams
    const dataStreamSources = resolveRes.data_streams.map(dataStream => {
      return {
        type: _onechatCommon.EsResourceType.dataStream,
        name: dataStream.name,
        indices: Array.isArray(dataStream.backing_indices) ? dataStream.backing_indices : [dataStream.backing_indices],
        timestamp_field: dataStream.timestamp_field
      };
    });

    // aliases
    const aliasSources = resolveRes.aliases.map(alias => {
      return {
        type: _onechatCommon.EsResourceType.alias,
        name: alias.name,
        indices: Array.isArray(alias.indices) ? alias.indices : [alias.indices]
      };
    });

    // indices
    const resolvedDataStreamNames = dataStreamSources.map(ds => ds.name);
    const resolvedAliasNames = aliasSources.map(alias => alias.name);
    const indexSources = resolveRes.indices.filter(index => {
      var _index$aliases;
      if (excludeIndicesRepresentedAsAlias && (_index$aliases = index.aliases) !== null && _index$aliases !== void 0 && _index$aliases.length && index.aliases.some(alias => resolvedAliasNames.includes(alias))) {
        return false;
      }
      if (excludeIndicesRepresentedAsDatastream && index.data_stream && resolvedDataStreamNames.includes(index.data_stream)) {
        return false;
      }
      return true;
    }).map(index => {
      return {
        type: _onechatCommon.EsResourceType.index,
        name: index.name
      };
    });
    const warnings = [];
    if (dataStreamSources.length > perTypeLimit) {
      warnings.push(`DataStreams results truncated to ${perTypeLimit} elements - Total result count was ${dataStreamSources.length}`);
    }
    if (aliasSources.length > perTypeLimit) {
      warnings.push(`Aliases results truncated to ${perTypeLimit} elements - Total result count was ${aliasSources.length}`);
    }
    if (indexSources.length > perTypeLimit) {
      warnings.push(`Indices results truncated to ${perTypeLimit} elements - Total result count was ${indexSources.length}`);
    }
    return {
      warnings,
      data_streams: (0, _lodash.take)(dataStreamSources, perTypeLimit),
      aliases: (0, _lodash.take)(aliasSources, perTypeLimit),
      indices: (0, _lodash.take)(indexSources, perTypeLimit)
    };
  } catch (e) {
    if ((0, _esErrors.isNotFoundError)(e)) {
      return {
        data_streams: [],
        aliases: [],
        indices: [],
        warnings: ['No sources found.']
      };
    }
    throw e;
  }
};
exports.listSearchSources = listSearchSources;