"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateFakeToolCallId = exports.extractToolReturn = exports.extractToolCalls = exports.extractTextContent = exports.createUserMessage = exports.createToolResultMessage = exports.createToolCallMessage = exports.createAIMessage = void 0;
var _uuid = require("uuid");
var _messages = require("@langchain/core/messages");
var _onechatServer = require("@kbn/onechat-server");
var _lodash = require("lodash");
var _prompts = require("../prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Extract the text content from a langchain message or chunk.
 */
const extractTextContent = message => {
  if (typeof message.content === 'string') {
    return message.content;
  } else {
    let content = '';
    for (const item of message.content) {
      if (item.type === 'text') {
        content += item.text;
      }
    }
    return content;
  }
};
exports.extractTextContent = extractTextContent;
/**
 * Extracts the tool calls from a message.
 */
const extractToolCalls = message => {
  if ((0, _messages.isAIMessage)(message)) {
    var _message$tool_calls$m, _message$tool_calls;
    return (_message$tool_calls$m = (_message$tool_calls = message.tool_calls) === null || _message$tool_calls === void 0 ? void 0 : _message$tool_calls.map(toolCall => {
      if (!toolCall.id) {
        throw new Error('Tool call must have an id');
      }
      return {
        toolCallId: toolCall.id,
        toolName: toolCall.name,
        args: toolCall.args
      };
    })) !== null && _message$tool_calls$m !== void 0 ? _message$tool_calls$m : [];
  }
  return [];
};

/**
 * Extract the structured tool return from a given tool message.
 * Note: this assumes the tool call was performed with the right configuration, so that
 * it was executed from a onechat agent.
 */
exports.extractToolCalls = extractToolCalls;
const extractToolReturn = message => {
  if (message.artifact) {
    if (!(0, _lodash.isArray)(message.artifact.results)) {
      throw new Error(`Artifact is not a structured tool artifact. Received artifact=${JSON.stringify(message.artifact)}`);
    }
    return message.artifact;
  } else {
    // langchain tool validation error (such as schema errors) are out of our control and don't emit artifacts...
    const content = extractTextContent(message);
    if (content.startsWith('Error:')) {
      return {
        results: [(0, _onechatServer.createErrorResult)(content)]
      };
    } else {
      throw new Error(`No artifact attached to tool message: ${JSON.stringify(message)}`);
    }
  }
};
exports.extractToolReturn = extractToolReturn;
const generateFakeToolCallId = () => {
  return (0, _uuid.v4)().substr(0, 6);
};
exports.generateFakeToolCallId = generateFakeToolCallId;
const createUserMessage = (content, {
  clean = false
} = {}) => {
  return new _messages.HumanMessage({
    content: clean ? (0, _prompts.cleanPrompt)(content) : content
  });
};
exports.createUserMessage = createUserMessage;
const createAIMessage = (content, {
  clean = false
} = {}) => {
  return new _messages.AIMessage({
    content: clean ? (0, _prompts.cleanPrompt)(content) : content
  });
};
exports.createAIMessage = createAIMessage;
const createToolResultMessage = ({
  content,
  toolCallId
}) => {
  return new _messages.ToolMessage({
    content: typeof content === 'string' ? content : JSON.stringify(content),
    tool_call_id: toolCallId
  });
};
exports.createToolResultMessage = createToolResultMessage;
const createToolCallMessage = (toolCallOrCalls, message) => {
  const toolCalls = (0, _lodash.isArray)(toolCallOrCalls) ? toolCallOrCalls : [toolCallOrCalls];
  return new _messages.AIMessage({
    content: message !== null && message !== void 0 ? message : '',
    tool_calls: toolCalls.map(toolCall => ({
      id: toolCall.toolCallId,
      name: toolCall.toolName,
      args: toolCall.args
    }))
  });
};
exports.createToolCallMessage = createToolCallMessage;