"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getToolHandler = getToolHandler;
var _get_logs_indices = require("../../utils/get_logs_indices");
var _get_metrics_indices = require("../../utils/get_metrics_indices");
var _get_typed_search = require("../../utils/get_typed_search");
var _time = require("../../utils/time");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_SERVICES_FROM_INDICES = 500;
async function getServicesFromLogsAndMetricsIndices({
  esClient,
  logsIndices,
  metricsIndices,
  start,
  end,
  environment,
  logger
}) {
  const allIndices = [...logsIndices, ...metricsIndices];
  if (allIndices.length === 0) {
    return [];
  }
  try {
    var _response$aggregation, _response$aggregation2, _response$aggregation3;
    const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
    const response = await search({
      index: allIndices,
      size: 0,
      track_total_hits: false,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: start,
                lte: end
              }
            }
          }, {
            exists: {
              field: 'service.name'
            }
          }, ...(environment ? [{
            term: {
              'service.environment': environment
            }
          }] : [])]
        }
      },
      aggs: {
        services: {
          terms: {
            field: 'service.name',
            size: MAX_SERVICES_FROM_INDICES
          },
          aggs: {
            environments: {
              terms: {
                field: 'service.environment',
                size: 10
              }
            }
          }
        }
      }
    });
    const buckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.services) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
    return buckets.map(bucket => {
      var _bucket$environments, _bucket$environments$, _bucket$environments$2;
      return {
        serviceName: bucket.key,
        environment: (_bucket$environments = bucket.environments) === null || _bucket$environments === void 0 ? void 0 : (_bucket$environments$ = _bucket$environments.buckets) === null || _bucket$environments$ === void 0 ? void 0 : (_bucket$environments$2 = _bucket$environments$[0]) === null || _bucket$environments$2 === void 0 ? void 0 : _bucket$environments$2.key
      };
    });
  } catch (error) {
    logger.debug(`Failed to get services from indices: ${error.message}`);
    return [];
  }
}
function mergeServices({
  apmServices,
  logsAndMetricsServices
}) {
  const serviceMap = new Map();
  for (const service of apmServices) {
    serviceMap.set(service.serviceName, service);
  }
  for (const service of logsAndMetricsServices) {
    if (!serviceMap.has(service.serviceName)) {
      serviceMap.set(service.serviceName, {
        serviceName: service.serviceName,
        environments: service.environment ? [service.environment] : undefined
      });
    }
  }
  return Array.from(serviceMap.values());
}
async function getToolHandler({
  core,
  plugins,
  request,
  esClient,
  dataRegistry,
  logger,
  start,
  end,
  environment,
  healthStatus
}) {
  var _apmResponse$items, _apmResponse$maxCount, _apmResponse$serviceO;
  const startMs = (0, _time.parseDatemath)(start);
  const endMs = (0, _time.parseDatemath)(end, {
    roundUp: true
  });
  const [logsIndices, metricsIndices] = await Promise.all([(0, _get_logs_indices.getLogsIndices)({
    core,
    logger
  }), (0, _get_metrics_indices.getMetricsIndices)({
    core,
    plugins,
    logger
  })]);
  const [apmResponse, logsAndMetricsServices] = await Promise.all([dataRegistry.getData('servicesItems', {
    request,
    environment,
    start,
    end
  }), getServicesFromLogsAndMetricsIndices({
    esClient,
    logsIndices,
    metricsIndices,
    start: startMs,
    end: endMs,
    environment,
    logger
  })]);
  const apmServices = (_apmResponse$items = apmResponse === null || apmResponse === void 0 ? void 0 : apmResponse.items) !== null && _apmResponse$items !== void 0 ? _apmResponse$items : [];

  // Filter APM services by health status (if provided) and convert latency to milliseconds
  const normalizedApmServices = apmServices.flatMap(service => {
    var _service$healthStatus;
    if (healthStatus && !healthStatus.includes((_service$healthStatus = service.healthStatus) !== null && _service$healthStatus !== void 0 ? _service$healthStatus : 'unknown')) {
      return [];
    }
    return [{
      ...service,
      latency: service.latency ? service.latency / 1000 : undefined
    }];
  });

  // Merge all services from different sources
  // When filtering by health status, exclude logs/metrics-only services since they don't have health data
  const services = mergeServices({
    apmServices: normalizedApmServices,
    logsAndMetricsServices: healthStatus ? [] : logsAndMetricsServices
  });
  return {
    services,
    maxCountExceeded: (_apmResponse$maxCount = apmResponse === null || apmResponse === void 0 ? void 0 : apmResponse.maxCountExceeded) !== null && _apmResponse$maxCount !== void 0 ? _apmResponse$maxCount : false,
    serviceOverflowCount: (_apmResponse$serviceO = apmResponse === null || apmResponse === void 0 ? void 0 : apmResponse.serviceOverflowCount) !== null && _apmResponse$serviceO !== void 0 ? _apmResponse$serviceO : 0
  };
}