"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_METRIC_CHANGE_POINTS_TOOL_ID = void 0;
exports.createGetMetricChangePointsTool = createGetMetricChangePointsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _get_metrics_indices = require("../../utils/get_metrics_indices");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_METRIC_CHANGE_POINTS_TOOL_ID = exports.OBSERVABILITY_GET_METRIC_CHANGE_POINTS_TOOL_ID = 'observability.get_metric_change_points';
const getMetricChangePointsSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchemaRequired,
  index: _zod.z.string().describe('The index or index pattern to find the metrics').optional(),
  kqlFilter: _zod.z.string().describe('A KQL filter to filter the metric documents, e.g.: my_field:foo').optional(),
  aggregation: _zod.z.object({
    field: _zod.z.string().describe(`Numeric field to aggregate and observe for changes (e.g., 'transaction.duration.us').`),
    type: _zod.z.enum(['avg', 'sum', 'min', 'max', 'p95', 'p99']).describe('Aggregation to apply to the specified field.')
  }).optional(),
  groupBy: _zod.z.array(_zod.z.string()).describe(`Optional keyword fields to break down metrics by to identify which specific group experienced a change.
Use only low-cardinality fields. Using many fields or high-cardinality fields can cause a large number of groups and severely impact performance.
Examples: ['service.name', 'service.version'], ['service.name', 'host.name'], ['cloud.availability_zone']
`).optional()
});
function createGetMetricChangePointsTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_METRIC_CHANGE_POINTS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Detects statistically significant changes in metrics across groups (e.g., by service, host, or custom fields).

When to use:
- Detecting significant changes in metric values (spike, dip, step change, trend change, distribution change, stationary/non‑stationary, indeterminable) and identifying when they occur.
`,
    schema: getMetricChangePointsSchema,
    tags: ['observability', 'metrics'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start,
      end,
      index,
      kqlFilter,
      aggregation,
      groupBy = []
    }, {
      esClient
    }) => {
      try {
        const metricIndexPatterns = await (0, _get_metrics_indices.getMetricsIndices)({
          core,
          plugins,
          logger
        });
        const topMetricChangePoints = await (0, _handler.getToolHandler)({
          esClient,
          start,
          end,
          index: index || metricIndexPatterns.join(','),
          kqlFilter,
          aggregation,
          groupBy
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              changePoints: topMetricChangePoints
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting metric change points: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Error getting metric change points: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}