"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getToolHandler = getToolHandler;
var _lodash = require("lodash");
var _get_change_points = require("../../utils/get_change_points");
var _time = require("../../utils/time");
var _dsl_filters = require("../../utils/dsl_filters");
var _get_typed_search = require("../../utils/get_typed_search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getMetricAggregation(aggregation) {
  if (!aggregation) {
    return {
      agg: {
        filter: {
          match_all: {}
        }
      },
      buckets_path: '_count'
    };
  }
  if (['min', 'max', 'sum', 'avg'].includes(aggregation.type)) {
    return {
      agg: {
        [aggregation.type]: {
          field: aggregation.field
        }
      },
      buckets_path: 'metric'
    };
  }
  const percentile = `${aggregation.type.split('p')[1]}.0`;
  return {
    agg: {
      percentiles: {
        field: aggregation.field,
        percents: [Number(percentile)],
        keyed: true
      }
    },
    buckets_path: 'metric'
  };
}
function getGroupingAggregation(groupingFields) {
  if (groupingFields.length === 0) {
    return {
      filters: {
        filters: [{
          match_all: {}
        }]
      }
    };
  }
  if (groupingFields.length === 1) {
    return {
      terms: {
        field: groupingFields[0]
      }
    };
  }
  return {
    multi_terms: {
      terms: groupingFields.map(groupingField => ({
        field: groupingField
      })),
      size: 10
    }
  };
}
async function getMetricChangePoints({
  index,
  start,
  end,
  kqlFilter: kuery,
  groupBy,
  aggregation,
  esClient
}) {
  var _response$aggregation, _response$aggregation2;
  const {
    agg: metricAgg,
    buckets_path: bucketsPathMetric
  } = getMetricAggregation(aggregation);
  const groupAgg = getGroupingAggregation(groupBy);
  const aggregations = {
    groups: {
      ...groupAgg,
      aggs: {
        time_series: {
          auto_date_histogram: {
            field: '@timestamp',
            buckets: 100
          },
          aggs: {
            metric: metricAgg,
            value: {
              bucket_script: {
                buckets_path: {
                  metric: bucketsPathMetric
                },
                script: 'params.metric'
              }
            }
          }
        },
        changes: {
          change_point: {
            buckets_path: 'time_series>value'
          }
          // elasticsearch@9.0.0 change_point aggregation is missing in the types: https://github.com/elastic/elasticsearch-specification/issues/3671
        }
      }
    }
  };
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
  const response = await search({
    index,
    size: 0,
    track_total_hits: false,
    query: {
      bool: {
        filter: [...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
          start: (0, _time.parseDatemath)(start),
          end: (0, _time.parseDatemath)(end)
        }), ...(0, _dsl_filters.kqlFilter)(kuery)]
      }
    },
    aggs: aggregations
  });
  const buckets = (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : (_response$aggregation2 = _response$aggregation.groups) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets;
  if (!buckets) {
    return [];
  }
  return await (0, _get_change_points.getChangePoints)({
    buckets
  });
}
async function getToolHandler({
  esClient,
  start,
  end,
  index,
  aggregation,
  kqlFilter: kqlFilterValue,
  groupBy
}) {
  const metricChangePoints = await getMetricChangePoints({
    esClient,
    index,
    start,
    end,
    aggregation,
    kqlFilter: kqlFilterValue,
    groupBy
  });
  return (0, _lodash.orderBy)(metricChangePoints.flat(), [item => item.changes.p_value]).slice(0, 25);
}