"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getToolHandler = getToolHandler;
var _lodash = require("lodash");
var _get_total_hits = require("../../utils/get_total_hits");
var _get_change_points = require("../../utils/get_change_points");
var _time = require("../../utils/time");
var _dsl_filters = require("../../utils/dsl_filters");
var _get_typed_search = require("../../utils/get_typed_search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getProbability(totalHits) {
  const probability = Math.min(1, 500_000 / totalHits);
  return probability > 0.5 ? 1 : probability;
}
async function getLogChangePoint({
  index,
  start,
  end,
  kqlFilter: kqlFilterValue,
  messageField,
  esClient
}) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3;
  const countDocumentsResponse = await esClient.asCurrentUser.search({
    size: 0,
    track_total_hits: true,
    index,
    query: {
      bool: {
        filter: [...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
          start: (0, _time.parseDatemath)(start),
          end: (0, _time.parseDatemath)(end)
        }), ...(0, _dsl_filters.kqlFilter)(kqlFilterValue)]
      }
    }
  });
  const totalHits = (0, _get_total_hits.getTotalHits)(countDocumentsResponse);
  if (totalHits === 0) {
    return [];
  }
  const aggregations = {
    sampler: {
      random_sampler: {
        probability: getProbability(totalHits)
      },
      aggs: {
        groups: {
          categorize_text: {
            field: messageField,
            size: 1000
          },
          aggs: {
            time_series: {
              auto_date_histogram: {
                field: '@timestamp',
                buckets: 100
              }
            },
            changes: {
              change_point: {
                buckets_path: 'time_series>_count'
              }
              // elasticsearch@9.0.0 change_point aggregation is missing in the types: https://github.com/elastic/elasticsearch-specification/issues/3671
            }
          }
        }
      }
    }
  };
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);
  const response = await search({
    index,
    size: 0,
    track_total_hits: false,
    query: {
      bool: {
        filter: [...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
          start: (0, _time.parseDatemath)(start),
          end: (0, _time.parseDatemath)(end)
        }), ...(0, _dsl_filters.kqlFilter)(kqlFilterValue)]
      }
    },
    aggs: aggregations
  });
  const buckets = (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : (_response$aggregation2 = _response$aggregation.sampler) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.groups) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.buckets;
  if (!buckets) {
    return [];
  }
  return await (0, _get_change_points.getChangePoints)({
    buckets: buckets
  });
}
async function getToolHandler({
  esClient,
  start,
  end,
  index,
  kqlFilter: kqlFilterValue,
  messageField
}) {
  const logChangePoints = await getLogChangePoint({
    index,
    esClient,
    start,
    end,
    kqlFilter: kqlFilterValue,
    messageField
  });
  return (0, _lodash.orderBy)(logChangePoints.flat(), [item => item.changes.p_value]).slice(0, 25);
}