"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLogCategories = getLogCategories;
exports.getToolHandler = getToolHandler;
var _get_logs_indices = require("../../utils/get_logs_indices");
var _get_typed_search = require("../../utils/get_typed_search");
var _get_total_hits = require("../../utils/get_total_hits");
var _dsl_filters = require("../../utils/dsl_filters");
var _time = require("../../utils/time");
var _ecs_otel_fields = require("../../utils/ecs_otel_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getToolHandler({
  core,
  logger,
  esClient,
  index,
  start,
  end,
  kqlFilter: kuery,
  fields,
  messageField
}) {
  var _index$split;
  const logsIndices = (_index$split = index === null || index === void 0 ? void 0 : index.split(',')) !== null && _index$split !== void 0 ? _index$split : await (0, _get_logs_indices.getLogsIndices)({
    core,
    logger
  });
  const boolFilters = [...(0, _dsl_filters.timeRangeFilter)('@timestamp', {
    start: (0, _time.parseDatemath)(start),
    end: (0, _time.parseDatemath)(end, {
      roundUp: true
    })
  }), ...(0, _dsl_filters.kqlFilter)(kuery), {
    exists: {
      field: messageField
    }
  }];
  const [highSeverityCategories, lowSeverityCategories] = await Promise.all([getLogCategories({
    esClient,
    logsIndices,
    boolQuery: {
      filter: boolFilters,
      must: [(0, _ecs_otel_fields.warningAndAboveLogFilter)()]
    },
    logger,
    categoryCount: 20,
    fields,
    messageField
  }), getLogCategories({
    esClient,
    logsIndices,
    boolQuery: {
      filter: boolFilters,
      must_not: [(0, _ecs_otel_fields.warningAndAboveLogFilter)()]
    },
    logger,
    categoryCount: 10,
    fields,
    messageField
  })]);
  return {
    highSeverityCategories,
    lowSeverityCategories
  };
}
async function getLogCategories({
  esClient,
  logsIndices,
  boolQuery,
  logger,
  categoryCount,
  fields,
  messageField
}) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5;
  const search = (0, _get_typed_search.getTypedSearch)(esClient.asCurrentUser);

  // Count total documents to determine sampling probability
  const countResponse = await search({
    index: logsIndices,
    size: 0,
    track_total_hits: true,
    query: {
      bool: boolQuery
    }
  });
  const totalHits = (0, _get_total_hits.getTotalHits)(countResponse);
  if (totalHits === 0) {
    logger.debug('No log documents found for the given query.');
    return undefined;
  }

  // Calculate sampling probability to get ~10,000 samples
  const targetSampleSize = 10000;
  const rawSamplingProbability = targetSampleSize / totalHits;
  const samplingProbability = rawSamplingProbability < 0.5 ? rawSamplingProbability : 1; // probability must be between 0.0 and 0.5 or exactly 1.0

  logger.debug(`Total log documents: ${totalHits}, using sampling probability: ${samplingProbability.toFixed(4)} using filter: ${JSON.stringify(boolQuery)}`);
  const response = await search({
    index: logsIndices,
    size: 0,
    track_total_hits: false,
    query: {
      bool: boolQuery
    },
    aggregations: {
      sampler: {
        random_sampler: {
          probability: samplingProbability,
          seed: 1
        },
        aggs: {
          categories: {
            categorize_text: {
              field: messageField,
              size: categoryCount,
              min_doc_count: 1
            },
            aggs: {
              sample: {
                top_hits: {
                  size: 1,
                  _source: false,
                  fields: [messageField, '@timestamp', ...fields],
                  sort: {
                    '@timestamp': {
                      order: 'desc'
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  const categories = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.sampler) === null || _response$aggregation3 === void 0 ? void 0 : (_response$aggregation4 = _response$aggregation3.categories) === null || _response$aggregation4 === void 0 ? void 0 : (_response$aggregation5 = _response$aggregation4.buckets) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.map(bucket => {
    var _bucket$sample, _bucket$sample$hits, _bucket$sample$hits$h, _bucket$sample$hits$h2;
    return {
      pattern: bucket.key,
      regex: bucket.regex,
      count: bucket.doc_count,
      sample: (_bucket$sample = bucket.sample) === null || _bucket$sample === void 0 ? void 0 : (_bucket$sample$hits = _bucket$sample.hits) === null || _bucket$sample$hits === void 0 ? void 0 : (_bucket$sample$hits$h = _bucket$sample$hits.hits) === null || _bucket$sample$hits$h === void 0 ? void 0 : (_bucket$sample$hits$h2 = _bucket$sample$hits$h[0]) === null || _bucket$sample$hits$h2 === void 0 ? void 0 : _bucket$sample$hits$h2.fields
    };
  })) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  return {
    categories,
    totalHits,
    samplingProbability
  };
}